/**
 * The WikiServlet handles the main interactions from the user and forwards to the
 * appropriate servlets and JSPs
 *
 * Copyright 2002 Gareth Cronin
 * This software is subject to the GNU Lesser General Public Licence (LGPL)
 */
package vqwiki.servlets;

import org.apache.log4j.Logger;
import vqwiki.ActionManager;
import vqwiki.Environment;
import vqwiki.PseudoTopicHandler;
import vqwiki.SearchEngine;
import vqwiki.SearchResultEntry;
import vqwiki.Topic;
import vqwiki.WikiAction;
import vqwiki.WikiBase;
import vqwiki.WikiException;
import vqwiki.utils.JSPUtils;
import vqwiki.utils.Utilities;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Vector;

/*
 * TODO This Servlet is a litle bit too complex. It could be a good idea to
 * refactorize and split resposibilities in different objects.
 */

public class WikiServlet extends VQWikiServlet {

  private static final Logger logger = Logger.getLogger(WikiServlet.class);

  // constants used as the action parameter in calls to this servlet
  public static final String ACTION_SEARCH = "action_search";
  public static final String ACTION_EDIT = "action_edit";
  public static final String ACTION_CANCEL = "Cancel";
  public static final String ACTION_ADMIN = "action_admin";
  public static final String ACTION_RSS = "RSS";
  public static final String ACTION_SAVE_USER = "action_save_user";
  public static final String ACTION_NOTIFY = "action_notify";
  public static final String ACTION_MEMBER = "action_member";

  /**
   * Servlet context
   */
  // why static? Better not to do that.
  private static ServletContext servletContext = null;

  private Map cachedContents = new HashMap();

  public void init(ServletConfig config) throws ServletException {
    super.init(config);

    /* store servlet context reference so that it can be accessed from a static
       context */
    servletContext = config.getServletContext();
  }

  /**
   * Returns a reference to the current servlet context
   */
  public static ServletContext getCurrentContext() {
    return servletContext;
  }

  protected void doPut(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse)
      throws ServletException, IOException {
    super.doPut(httpServletRequest, httpServletResponse);
    Environment.getInstance().setRealPath(httpServletRequest.getSession().getServletContext().getRealPath("/"));
  }

  /**
   * GET requests should come from topic display requests, edit requests, diffs
   */
  protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

    String characterEncoding = Utilities.extractCharacterEncoding(request, response);
  	
    logger.debug("GET Topic: " + request.getQueryString());
    Environment.getInstance().setRealPath(request.getSession().getServletContext().getRealPath("/"));
    

    if (!checkAccess(request)) {
    	logAccess(request, "Access denied!");
        request.setAttribute("exception", Utilities.resource("error.forbidden.text", request.getLocale(), "Forbidden"));
        request.setAttribute("title", Utilities.resource("error.forbidden.title", request.getLocale(), "Forbidden"));
        request.setAttribute("javax.servlet.jsp.jspException", null);
        dispatch("/jsp/servlet-error.jsp", request, response);
        return;
    } else {
        logAccess(request);    	
    }

    int display = WikiBase.DISPLAY;
    String displayAdd="";
    if (request.getParameter("display")!= null) {
    	String displayStr = request.getParameter("display");
    	if (displayStr.equalsIgnoreCase("screen")) {display = WikiBase.DISPLAY; displayAdd = "&display=screen";}
    	if (displayStr.equalsIgnoreCase("print")) {display = WikiBase.PRINTPREV; displayAdd = "&display=print";}
    	if (displayStr.equalsIgnoreCase("export")) {display = WikiBase.EXPORTPREV; displayAdd = "&display=export";}
    }
    
    Environment en = Environment.getInstance();  
    en.setRealPath(request.getSession().getServletContext().getRealPath("/"));

//  request.setAttribute("lastRequest", HttpUtils.getRequestURL(request));
    request.setAttribute("lastRequest", request.getRequestURL());

    // expire now
    response.setDateHeader("Expires", System.currentTimeMillis() - 24 * 60 * 60 * 1000);

    WikiBase wb = null;
    try {
    	wb = WikiBase.getInstance();
    } catch (Exception e) {
    	throw new ServletException ("Could not get WikiBase-Instance");
    }
    
    String virtualWiki = null;
    try {
      virtualWiki = Utilities.extractVirtualWiki(request);
    } catch (WikiException e) {
        request.setAttribute("exception", e.getMessage());
        request.setAttribute("title", "Error");
        request.setAttribute("javax.servlet.jsp.jspException", e);
        dispatch("/jsp/servlet-error.jsp", request, response);
        return;
    }
    logger.debug("virtual wiki: " + virtualWiki);
    request.setAttribute("virtual-wiki", virtualWiki);
    request.setAttribute("virtualWiki", virtualWiki);

    buildLayout(request, virtualWiki, display);

    String topic = request.getQueryString();
    if ((topic == null) || (topic.equals(""))){
      topic = wb.getVirtualWikiHome(virtualWiki);
    }

    if (topic.indexOf('&') > 0) {
      topic = topic.substring(0, topic.indexOf('&'));
    }    
    topic = JSPUtils.decodeURL(topic, characterEncoding);
    topic = topic.trim();
   
    String topicTest= topic;
    if (topic.startsWith("topic=")) {
    	topicTest = topic.substring(6);
    }
    if (!topic.startsWith("action=")&&!topic.startsWith("function=")) {

	    if (!Utilities.checkURLChars(topicTest)) {
	    	WikiServletException err = new WikiServletException ("Wrong character(s) in topic: " + Utilities.getWrongURLChars(topicTest));
	        request.setAttribute("exception", err);
	        request.setAttribute("title", "Error");
	        request.setAttribute("javax.servlet.jsp.jspException", err);
	        dispatch("/jsp/servlet-error.jsp", request, response);
	        return;
	    }

    }    
    request.setAttribute("env", en);
    if (en.getStringSetting(Environment.PROPERTY_BASE_CONTEXT) == null) {
      try {
//        StringBuffer originalURL = HttpUtils.getRequestURL(request);
          StringBuffer originalURL = request.getRequestURL();

        logger.info(originalURL);

        String baseContext = originalURL.substring(0, originalURL.length() - virtualWiki.length() - 6);
        en.setSetting(Environment.PROPERTY_BASE_CONTEXT, baseContext);
      }
      catch (Exception e) {
        error(request, response, e);
        return;
      }
    }

    RequestDispatcher dispatch;
    // make decision on action if one exists
    String action = request.getParameter("action");
    logger.debug("action: " + action);
    if (action != null) {
      String actionRedirect = PseudoTopicHandler.getInstance().getRedirectURL(action);
      if (action.equals(ACTION_EDIT)) {
        // a request to edit a topic
        if (Environment.getInstance().getForceUsername()) {
          if (Utilities.getUserFromRequest(request) == null) {
            dispatch = request.getRequestDispatcher("/jsp/createUser.jsp");
            dispatch.forward(request, response);
            return;
          }
        }
        dispatch = request.getRequestDispatcher("/EditServlet");
        dispatch.forward(request, response);
        return;
      } else if (action.equals(ACTION_RSS)) {
        dispatch = request.getRequestDispatcher("/RSS");
        dispatch.forward(request, response);
        return;
      } else if (actionRedirect != null) {
        logger.debug("Using redirect from pseudotopics actions: " + actionRedirect);
        dispatch(actionRedirect, request, response);
        return;
      } else if (ActionManager.getInstance().actionExists(action)) {
        logger.debug("using action mapping from ActionManager");
        try {
          WikiAction wikiAction = ActionManager.getInstance().getActionInstance(action);
          if (wikiAction != null) {
            wikiAction.doAction(request, response);
            return;
          }
        } catch (Exception e) {
          logger.error("error running action", e);
          request.setAttribute("exception", e);
          request.setAttribute("title", "Error");
          log("Error in " + this.getClass(), e);
          if (e instanceof WikiServletException) {
            request.setAttribute("javax.servlet.jsp.jspException", e);
          }
          dispatch("/jsp/servlet-error.jsp", request, response);
          return;
        }
      }
    }

    logger.debug("no action mappings, assuming topic");


    request.setAttribute("topic", topic);
    if (topic.equals(wb.getVirtualWikiHome(virtualWiki))) {
    	request.setAttribute("title", wb.getVirtualWikiHomeTitle(virtualWiki));
    } else {
        request.setAttribute("title", topic);    	
    }
    
    // make decision based on topic
    response.setContentType("text/html");

    String pseudotopicRedirect = PseudoTopicHandler.getInstance().getRedirectURL(topic);


    if (pseudotopicRedirect != null) {
      PseudoTopicHandler.getInstance().setAttributes(topic, request);
      dispatch(pseudotopicRedirect, request, response);
      return;
    }
    else {
      dispatchTopic(request, topic, virtualWiki, response, en);
    }
  }

  private void buildLayout(HttpServletRequest request, String virtualWiki, int display) {
    // build the layout contents
    ResourceBundle messages = getMessages(request.getLocale());
  	WikiBase wb = null;
  	try {
  		wb = WikiBase.getInstance();
  	} catch (Exception e) {};

	try {
	    addIfNotEmpty(
	        request, "leftMenu", 
//	        getCachedContent(virtualWiki, messages.getString("specialpages.leftMenu"))
	    	wb.readCooked(virtualWiki, messages.getString("specialpages.leftMenu"), display)
	    );
    } catch (Exception e) {}
    
    try {
	    request.setAttribute(
	        "topArea", 
//          getCachedContent(virtualWiki, messages.getString("specialpages.topArea"))
	    	wb.readCooked(virtualWiki, messages.getString("specialpages.topArea"), display)
	    );
    } catch (Exception e) {}

    try {
	    request.setAttribute(
	        "bottomArea", 
//          getCachedContent(virtualWiki, messages.getString("specialpages.bottomArea"))
	    	wb.readCooked(virtualWiki, messages.getString("specialpages.bottomArea"), display)
	    );
    } catch (Exception e) {}

    request.setAttribute(
        "StyleSheet", 
        getCachedRawContent(virtualWiki, messages.getString("specialpages.stylesheet"))
    );
  }

  private ResourceBundle getMessages(Locale locale) {
    ResourceBundle messages = Utilities.getMessages(locale);
    return messages;
  }

  /**
   * Setup request and despatch to the JSP to display a topic
   * @param request
   * @param topic
   * @param virtualWiki
   * @param response
   * @param en
   */
  private void dispatchTopic(HttpServletRequest request, String topic, String virtualWiki,
                             HttpServletResponse response, Environment en) {

  	String characterEncoding = Utilities.extractCharacterEncoding(request, response, false);
  	WikiBase wb = null;
  	try {
  		wb = WikiBase.getInstance();
  	} catch (Exception e) {};
    try {
      if (wb.isAdminOnlyTopic(request.getLocale(), virtualWiki, topic)) {
        if (!Utilities.isAdmin(request)) {
          request.setAttribute("title", Utilities.resource("login.title", request.getLocale(), "WikiLogin"));
//        logger.debug("Current URL: " + HttpUtils.getRequestURL(request));
          logger.debug("Current URL: " + request.getRequestURL());
          String rootPath = JSPUtils.createRootPath(request, virtualWiki, false);
          StringBuffer buffer = new StringBuffer();
          buffer.append(rootPath);
          buffer.append("Wiki?" + topic);
          request.setAttribute(
              "redirect",
              buffer.toString()
          );
          dispatch("/jsp/login.jsp", request, response);
          return;
        }
      }
    }
    catch (Exception e) {
      logger.error("error checking admin only topic", e);
    }

    int display = WikiBase.DISPLAY;
    String displayAdd="";
    if (request.getParameter("display")!= null) {
    	String displayStr = request.getParameter("display");
    	if (displayStr.equalsIgnoreCase("screen")) {display = WikiBase.DISPLAY; displayAdd = "&display=screen";}
    	if (displayStr.equalsIgnoreCase("print")) {display = WikiBase.PRINTPREV; displayAdd = "&display=print";}
    	if (displayStr.equalsIgnoreCase("export")) {display = WikiBase.EXPORTPREV; displayAdd = "&display=export";}
    }

    String contents = null;
    boolean finished = true;
    do {
    	try {
    		contents = wb.readCooked(virtualWiki, topic, display);
    	} catch (Exception e) {
    		error(request, response, e);
    		return;
    	}
    } while (!finished);

    // deal with redirection
    if (contents.startsWith("redirect:") && (contents.trim().length() > 9)) {
    	String redirTopic = null;
    	String redirContent = contents;
        if (redirContent.indexOf("<br/>") >= 0) {
        	redirContent = redirContent.substring(0, redirContent.indexOf("<br/>")).trim();
        }
    	if (redirContent.startsWith("redirect:\"")) {
        	int first = redirContent.indexOf("\"");
       		int last = redirContent.lastIndexOf("\"");
        	if (first != last) {
        		redirTopic=redirContent.substring (first + 1, last);
        	} else {
        		redirTopic=null;
        	}    		
    	} else if (redirContent.trim().indexOf(" ") < 0) { // watch out for case where a space has been left
    		redirTopic = redirContent.substring(redirContent.indexOf(":") + 1).trim();
    	} else {
    		redirTopic = redirContent.substring(redirContent.indexOf(":") + 1, redirContent.indexOf(" ")).trim();    		
    	}
    	if (redirTopic != null) {
    		redirect("Wiki?" + JSPUtils.encodeURL(redirTopic, characterEncoding) + displayAdd, response);
    		return;
    	}
    }
    // highlight search result
    if (request.getParameter("highlight") != null) {
      String highlightparam = request.getParameter("highlight");
      String highlighttext = "<b style=\"color:black;background-color:#ffff66\">###</b>";
      highlighttext = "<span class=\"searchhighlight\">###</span>";
      String placeholder = "###";
      contents = markToReplaceOutsideHTML(contents, highlightparam);
      contents = replaceMarked(contents, highlightparam, highlighttext, placeholder);      
    }
    
    // -------------
    // Handle page history (breadcrumb trail)
    HttpSession session = request.getSession();
    Vector history = (Vector) session.getAttribute("history");
    String historyVirtualWiki = (String) session.getAttribute("historyVirtualWiki");
    if (historyVirtualWiki != null) {
      if (!virtualWiki.equals(historyVirtualWiki)) {
        // reset history on virtual wiki changes
        history = new Vector();
        session.setAttribute("historyVirtualWiki", virtualWiki);
      }
    }
    else {
      session.setAttribute("historyVirtualWiki", virtualWiki);
    }
    if (history == null) {
      history = new Vector();
    }
    // if user clicked on "refresh"
    if (history.size() > 0 && topic.equals(history.lastElement())) {
      history.remove(history.size() - 1);
    }
    // add current page to history
    if (history.contains(topic)) {
      // go back in history
      int found = history.indexOf(topic);
      int pos = history.size() - 1;
      while (pos >= found) {
        history.remove(pos);
        pos--;
      }
    }
    // store the history in the request
    request.setAttribute("historyThisPage", new Vector(history));

    // really add it
    history.add(topic);

    // store it in session
    session.setAttribute("history", history);

    List ignoreTheseTopicsList = new ArrayList();
    
    ignoreTheseTopicsList.add("LeftMenu");
    ignoreTheseTopicsList.add("BottomArea");
    ignoreTheseTopicsList.add("TopArea");
    
    try {
      SearchEngine sedb = wb.getSearchEngineInstance();
      int maxBackLinks = en.getIntSetting(Environment.PROPERTY_MAXIMUM_BACKLINKS);

      // create backlinks
      if (topic != null) {
        Collection results = sedb.findLinkedTo(virtualWiki, topic, display);

        ResourceBundle messages = getMessages(request.getLocale());

        if (results != null && results.size() > 0) {
          StringBuffer buffer = new StringBuffer("");
          buffer.append("<br/><br/><span class=\"backlinks\">");
          buffer.append("\"");
          buffer.append(topic);
          buffer.append("\" ");
          buffer.append(messages.getString("topic.ismentionedon"));
          buffer.append(" ");
          Iterator it = results.iterator();
          String divider = "";
          int count = 0;
          for (; count < maxBackLinks && it.hasNext(); ) {
            SearchResultEntry result = (SearchResultEntry) it.next();
            String pathRoot = JSPUtils.createRootPath(request, virtualWiki, false);
            request.setAttribute("pathRoot", pathRoot);
            if ((!result.getTopic().equals(topic)) && (!ignoreTheseTopicsList.contains(result.getTopic()))) {
              count++;
              buffer.append(divider);
              buffer.append("<a href=\"");
              buffer.append(pathRoot);
              buffer.append("Wiki?");
              buffer.append(JSPUtils.encodeURL(result.getTopic(), characterEncoding));
              buffer.append("&highlight=");
              buffer.append(JSPUtils.encodeURL(topic, characterEncoding));
              buffer.append("\">");
              buffer.append(result.getTopic());
              buffer.append("</a>");
              divider = " | ";
            }
          }
          if (count == 20) {
            buffer.append("...");
          }
          buffer.append("</span>");

          if (count > 0) {
          	contents += buffer.toString();
          }
        }
      }
    }
    catch (Exception e) {
      logger.warn(e.getMessage(), e);
    }

    request.setAttribute("contents", contents);
    logger.debug("contents: " + contents);
    if (Environment.getInstance().isVersioningOn()) {
      try {
        logger.debug("topic: " + topic);
        Topic topicData = new Topic(topic);
        java.util.Date revDate = topicData.getMostRecentRevisionDate(virtualWiki);
        String author = topicData.getMostRecentAuthor(virtualWiki);
        if (revDate != null) {
          request.setAttribute("lastRevisionDate", revDate);
          if (author != null) {
            request.setAttribute("lastAuthor", author);
          }
        }
      }
      catch (Exception e) {
        logger.warn(e.getMessage(), e);
      }
    }

    boolean readOnly = false;
    try {
      if (topic != null) {
        Topic t = new Topic(topic);
        readOnly = t.isReadOnlyTopic(virtualWiki);
      }
    }
    catch (Exception e) {
      logger.warn(e.getMessage(), e);
    }
    request.setAttribute("readOnly", new Boolean(readOnly));

    if (topic.equals(wb.getVirtualWikiHome(virtualWiki))) {
    	dispatch("/jsp/wikiEntry.jsp", request, response);
    } else {
    	dispatch("/jsp/wiki.jsp", request, response);    	
    }
  }

  private void addIfNotEmpty(HttpServletRequest request, String name, String content) {
    logger.debug("addIfNotEmpty called for " + name + "/" + content);
    if (content == null) {
      logger.debug("content provided is null, returning");
      return;
    }
    content = content.trim();
    if ("".equals(content)) {
      logger.debug("content is empty, returning");
      return;
    }
    if ("delete".equalsIgnoreCase(content)) {
      logger.debug("content is marked for purging, returning");
      return;
    }
    if ("This is a new topic".equalsIgnoreCase(content)) {
      logger.debug("topic is an unchanged new topic, returning");
      return;
    }
    logger.debug("setting content " + name + " = " + content);
    request.setAttribute(name, content);
  }

  private interface WikiReader {

    String read(String virtualWiki, String topic) throws Exception;
  }

  private static WikiReader cookedReader = new WikiReader() {
    public String read(String virtualWiki, String topic) throws Exception {
      return WikiBase.getInstance().readCooked(virtualWiki, topic);
    }
  };

  private static WikiReader rawReader = new WikiReader() {
    public String read(String virtualWiki, String topic) throws Exception {
      return WikiBase.getInstance().readRaw(virtualWiki, topic);
    }
  };

  private String getCached(String virtualWiki, String topic, WikiReader wr) {
    String content = (String) cachedContents.get(virtualWiki + "-" + topic);
    if (content == null) {
      try {
        logger.debug("reloading topic " + topic);
        content = wr.read(virtualWiki, topic);
        synchronized (cachedContents) {
          cachedContents.put(virtualWiki + "-" + topic, content);
        }
      }
      catch (Exception e) {
        logger.warn("error getting cached page", e);
        return null;
      }
    }
    return content;
  }

  /**
   * Implements a caching for "fixed" contents.
   * Caching is particularly important for often-asked topics (like left-side-menu,
   * top-banners, bottom-banners, etc. that are asked for every request)
   *
   * @param virtualWiki the virtual wiki
   * @param topic       the topic name
   * @return the topic contents
   */
  private String getCachedContent(String virtualWiki, String topic) {
    return getCached(virtualWiki, topic, cookedReader);
  }

  /**
   * Implements a caching for "fixed" contents.
   * Caching is particularly important for often-asked topics (like left-side-menu,
   * top-banners, bottom-banners, etc. that are asked for every request)
   *
   * @param virtualWiki the virtual wiki
   * @param topic       the topic name
   * @return the topic contents
   */
  private String getCachedRawContent(String virtualWiki, String topic) {
    return getCached(virtualWiki, topic, rawReader);
  }


  /**
   * Clears the cached content
   * This method is called when a "edit-save" or "edit-cancel" is invoked.
   * <p/>
   * Clearing all cached contents forces to reload.
   */
  private void removeCachedContents() {
    if (logger.isDebugEnabled()) {
      logger.debug(
          "Removing Cached Contents; " +
          "cachedContents.size() = " + cachedContents.size()
      );
    }
    cachedContents.clear();
  }


  /**
   * POST requests should only come from saves and searches
   */
  protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

  	WikiBase wb = null;
  	try {
  		wb = WikiBase.getInstance();
  	} catch (Exception e) {};
  	logger.debug("POST");
//  request.setAttribute("lastRequest", HttpUtils.getRequestURL(request));
    request.setAttribute("lastRequest", request.getRequestURL());
    RequestDispatcher dispatch;
    response.setContentType("text/html");
    String virtualWiki = null;
    try {
      virtualWiki = Utilities.extractVirtualWiki(request);
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    logger.debug("virtual wiki: " + virtualWiki);
    //  TODO virtual-wiki and virtualWiki request params should be unified.
    request.setAttribute("virtual-wiki", virtualWiki);
    request.setAttribute("virtualWiki", virtualWiki);
    request.setAttribute("env", Environment.getInstance());
    
    int display = WikiBase.DISPLAY;
    String displayAdd="";
    if (request.getParameter("display")!= null) {
    	String displayStr = request.getParameter("display");
    	if (displayStr.equalsIgnoreCase("screen")) {display = WikiBase.DISPLAY; displayAdd = "&display=screen";}
    	if (displayStr.equalsIgnoreCase("print")) {display = WikiBase.PRINTPREV; displayAdd = "&display=print";}
    	if (displayStr.equalsIgnoreCase("export")) {display = WikiBase.EXPORTPREV; displayAdd = "&display=export";}
    }
        
    buildLayout(request, virtualWiki, display);

    // make decision based on action
    String action = request.getParameter("action");
    if (action != null) {
      logger.debug("Wiki action: " + action);
      String actionRedirect = PseudoTopicHandler.getInstance().getRedirectURL(action);
      logger.debug("actionRedirect: " + actionRedirect);
      if (actionRedirect != null) {
        /*
         * I handle the layout pages in a cache: I do not reload the for every
         * request, I just reload when an update is made on any page (SaveTopicServlet)
         * This is for performance reasons.
         *
         * In the previous version I added removeCachedContents() call in update
         * metods; in this version, this didn't work due to this new
         * implementation  by PseudoTopicHandler, so I needed to add
         * checkActionAndRemoveCachedContentsIfNeeded method call.
         *
         * TODO: Simplify this servlet.
         *
         */
        checkActionAndRemoveCachedContentsIfNeeded(action, request.getLocale());
        logger.debug("Using redirect from pseudotopics actions: " + actionRedirect);
        dispatch(actionRedirect, request, response);
        return;
      }
      else if (action.equals(ACTION_SEARCH)) {
        // a search request has been made
        dispatch = request.getRequestDispatcher("/SearchServlet");
        dispatch.forward(request, response);
        return;
      }
      else if (checkAction(action, Utilities.resource("edit.action.save", request.getLocale(), "Save"))) {
        // a save request has been made
        logger.debug("Dispatching save");
        removeCachedContents();
        dispatch = request.getRequestDispatcher("/SaveTopicServlet");
        dispatch.forward(request, response);
        return;
      }
      else if (checkAction(action, Utilities.resource("edit.action.append", request.getLocale(), "Append"))) {
        // a save request has been made
        logger.debug("Dispatching append template");
        dispatch = request.getRequestDispatcher("/SaveTopicServlet");
        dispatch.forward(request, response);
        return;
      }
      else if (action.equals(ACTION_ADMIN)) {
        // request to save admin values
        logger.debug("Despatching admin servlet");
        dispatch = request.getRequestDispatcher("/AdministrationServlet");
        dispatch.forward(request, response);
        return;
      }
      else if (checkAction(action, Utilities.resource("edit.action.savetemplate", request.getLocale(), "Save as Template"))) {
        // save template
        logger.debug("Despatching save template");
        removeCachedContents();
        dispatch = request.getRequestDispatcher("/SaveTemplateServlet");
        dispatch.forward(request, response);
        return;
      }
      else if (action.equals(ACTION_RSS)) {
        // save template
        dispatch = request.getRequestDispatcher("/RSS");
        dispatch.forward(request, response);
        return;
      }
      else if (action.equals(Utilities.resource("edit.action.cancel", request.getLocale(), "Cancel"))) {
        // cancellation of edit
        String topic = request.getParameter("topic");
        /*
        try {
          WikiBase.getInstance().unlockTopic(virtualWiki, Utilities.decodeSafeURL(topic));
        }
        catch (Exception err) {
          error(request, response, new WikiServletException(err.getMessage()));
          return;
        }
        */
        String next = "Wiki?" + topic;
        removeCachedContents();
        response.sendRedirect(response.encodeRedirectURL(next));
        return;
      }
      else if (action.equals(ACTION_SAVE_USER)) {
        String username = request.getParameter("username");
        Cookie c = Utilities.createUsernameCookie(username);
        logger.debug("Delivering cookie: " + c.getName() + " " + c.getValue());
        try {
          response.addCookie(c);
        }
        catch (Exception e) {
          logger.warn(e.getMessage(), e);
          ResourceBundle messages =
              getMessages(request.getLocale());

          error(request, response, new WikiServletException(messages.getString("exception.badusername")));
          return;
        }
        String topic = request.getParameter("topic");
        request.setAttribute("title", "SetUsername");
        if (topic == null || "".equals(topic)) {
          dispatch = request.getRequestDispatcher("/jsp/createUser.jsp");
          dispatch.forward(request, response);
          return;
        }
        else {
          request.setAttribute("topic", topic);
          dispatch = request.getRequestDispatcher("/EditServlet");
          dispatch.forward(request, response);
          return;
        }
      }
      else if (action.equals(ACTION_NOTIFY)) {
        dispatch = request.getRequestDispatcher("/NotifyServlet");
        dispatch.forward(request, response);
        return;
      }
      else if (action.equals(ACTION_MEMBER)) {
        dispatch = request.getRequestDispatcher("/MemberServlet");
        dispatch.forward(request, response);
        return;
      }
      // in case, no action can be dispatched, go to a normal wiki page
    }
    String topic = request.getParameter("topic");
    if (topic.equals(wb.getVirtualWikiHome(virtualWiki))) {
        dispatch = request.getRequestDispatcher("/jsp/wikiEntry.jsp");
    } else {
        dispatch = request.getRequestDispatcher("/jsp/wiki.jsp");
    }
    
    dispatch = request.getRequestDispatcher("/jsp/wiki.jsp");
    dispatch.forward(request, response);
  }

  /**
   * if the action is Save, SaveTemplate or CancelFromEdit, clears the cache
   * and force the wiki to reload layout page elements
   * (topArea, bottomArea, leftMenu, etc.)
   */
  private void checkActionAndRemoveCachedContentsIfNeeded(String action, Locale locale) {
    if ((checkAction(action, Utilities.resource("edit.action.save", locale, "Save"))) ||
        (checkAction(action, Utilities.resource("edit.action.savetemplate", locale, "Save as Template"))) ||
        (checkAction(action, Utilities.resource("edit.action.cancel", locale, "Cancel")))
    ) {
      removeCachedContents();
    }
  }

  private boolean checkAction(String original, String test) {
    if (original.equals(test)) {
      return true;
    }

    // find in test the first character, which is not alpabetic
    int count;
    String test2 = test.toLowerCase();
    for (count = 0; count < test.length(); count++) {
      if (!(test2.charAt(count) >= 'a' && test2.charAt(count) <= 'z')) {
        break;
      }
    }
    count--;
    if (count < 0) {
      return false;
    }
    // fix by pcs_org
    if (original.length() < count) {
      return false;
    }
    if (original.substring(0, count).equals(test.substring(0, count))) {
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Mark all needles in a haystack, so that they can be replaced later. Take special care on HTML,
   * so that no needle is replaced inside a HTML tag.
   *
   * @param haystack The haystack to go through.
   * @param needle   The needle to search.
   * @return The haystack with all needles (outside HTML) marked with the char \u0000
   */
  public static String markToReplaceOutsideHTML(String haystack, String needle) {
    if (needle.length() == 0) {
      return haystack;
    }

    StringBuffer sb = new StringBuffer();
    boolean inHTMLmode = false;
    int l = haystack.length();
    for (int j = 0; j < l; j++) {
      char c = haystack.charAt(j);
      switch (c) {
        case '<':
          if (((j + 1) < l) && (haystack.charAt(j + 1) != ' ')) {
            inHTMLmode = true;
          }
          break;
        case '>':
          if (inHTMLmode) {
            inHTMLmode = false;
          }
          break;
      }
      if ((c == needle.charAt(0) || Math.abs(c - needle.charAt(0)) == 32) &&
          !inHTMLmode) {
        boolean ok = true;
        if ((j + needle.length()) > l ||
            !haystack.substring(j, j + needle.length()).equalsIgnoreCase(needle)) {
          ok = false;
        }

        if (ok) {
          sb.append('\u0000');
          for (int k = 0; k < needle.length(); k++) {
            sb.append(haystack.charAt(j + k));
          }
          j = j + needle.length() - 1;
        }
        else {
          sb.append(c);
        }
      }
      else {
        sb.append(c);
      }
    }
    return sb.toString();
  }

  
  public static String replaceMarked(String text, String subOriginal, String highlighText, String placeholder) {
  	int length = subOriginal.length();
  	String v1 = subOriginal;
  	String v2 = subOriginal.substring(0, 1).toUpperCase()+subOriginal.substring(1);
  	while (text.indexOf('\u0000')>=0) {
  		int pos = text.indexOf('\u0000');
  		String substitution = text.substring(pos+1, pos + length + 1);
  		String highlightThis = Utilities.replaceString(highlighText, placeholder, substitution);
  		if (substitution.equals(v1) | substitution.equals(v2)) {
  			text = Utilities.replaceString(text, '\u0000'+substitution, highlightThis);
  		} else {
  			text = Utilities.replaceString(text, '\u0000'+substitution, substitution);  			
  		}
  	}
  	return text;
  }
  /**
   * Replace all needles inside the text with their replacements.
   *
   * @param text        The text or haystack, where all needles are already marked with the unicode character \u0000
   * @param needle      The needle to search
   * @param replacement The text, which replaces the needle
   * @return String containing the text with the needle replaced by the replacement.
   */
  public static String replaceMarked(String text, String needle, String replacement) {

  	needle = '\u0000' + needle;

    text = Utilities.replaceString(text, needle, replacement);
        
    return text;
  }

}

