/*
 * Copyright 2002 Gareth Cronin
 * This software is subject to the GNU Public Licence
 */
package vqwiki.servlets;

import java.io.IOException;
import java.util.Collection;
import java.util.ResourceBundle;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import vqwiki.Change;
import vqwiki.ChangeLog;
import vqwiki.SearchEngine;
import vqwiki.Topic;
import vqwiki.WikiBase;
import vqwiki.file.FileChangeLog;
import vqwiki.utils.JSPUtils;
import vqwiki.utils.Utilities;
import org.apache.log4j.Logger;

public class SaveTopicServlet extends VQWikiServlet {
  private static final Logger logger = Logger.getLogger(SaveTopicServlet.class);
  protected final static String NO_TEMPLATE = "--No template--";

  protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
  	String requestCharacterEncoding = Utilities.extractCharacterEncoding(request, response);
  	String responseCharacterEncoding = Utilities.extractCharacterEncoding(response);
  	
    logger.debug("Request for save");
    String virtualWiki = null;
    String user = request.getRemoteAddr();
    logger.debug("user: " + user);
    if (Utilities.getUserFromRequest(request) != null)
      user = Utilities.getUserFromRequest(request);

    String topic = request.getParameter("topic");
    logger.debug("Saving topic: " + topic);

    if (topic == null) {
      WikiServletException err = new WikiServletException("Topic must be specified");
      error(request, response, err);
      return;
    }
   
    topic = JSPUtils.decodeURL(topic, requestCharacterEncoding);
    
    WikiBase base = null;
    try {
      base = WikiBase.getInstance();
      virtualWiki = (String) request.getAttribute("virtual-wiki");
      logger.debug("vwiki is " + virtualWiki);
    }
    catch (Exception err) {
      error(request, response, new WikiServletException(err.getMessage()));
      return;
    }

    ResourceBundle messages = Utilities.getMessages(request.getLocale());

    String append = messages.getString("edit.action.append");
    if (append == null) append = "append";
    if (request.getParameter("template") != null && append.equals(request.getParameter("action"))) {
      logger.debug("appending template contents from template: " + request.getParameter("template"));
      if (!request.getParameter("template").equals(NO_TEMPLATE)) {
        String templateContents = null;
        Collection templateNames = null;
        try {
          templateContents = base.getTemplate(virtualWiki, request.getParameter("template"));
          templateNames = base.getTemplates(virtualWiki);
        }
        catch (Exception e) {
          error(request, response, e);
          return;
        }
        StringBuffer buffer = new StringBuffer();
        buffer.append(request.getParameter("contents"));
        buffer.append(templateContents);
        request.setAttribute("contents", buffer.toString());
        String title = "Edit " + topic;
        request.setAttribute("title", title);
        request.setAttribute("templateNames", templateNames);
        request.setAttribute("topic", topic);
        dispatch("/jsp/edit.jsp", request, response);
        return;
      }
    }


    try {
      Topic t = new Topic(topic);
      if (t.isReadOnlyTopic(virtualWiki))
        throw new WikiServletException(WikiServletException.READ_ONLY);

      String key = request.getSession().getId();

      if (!base.holdsLock(virtualWiki, topic, key)) {
        logger.debug("No lock for " + virtualWiki + "/" + topic + "/" + key);
        error(request, response, new WikiServletException(WikiServletException.LOCK_TIMEOUT));
        return;
      }

      String contents = request.getParameter("contents");
      if (contents == null) {
   		error(request, response, new WikiServletException("Contents must be supplied"));
   		return;
      }

      base.write(virtualWiki, contents, request.getParameter("convertTabs") != null, topic, user);
      
      if (request.getParameter("minorEdit") == null){
        Change change = new Change();
        change.setTopic(topic);
        change.setUser(user);
        change.setTime(new java.util.Date());
        change.setVirtualWiki(virtualWiki);
  
        ChangeLog cl = WikiBase.getInstance().getChangeLogInstance();
  
        cl.logChange(change, request);
      }
      
      SearchEngine sedb = WikiBase.getInstance().getSearchEngineInstance();

      sedb.indexText(virtualWiki, topic, request.getParameter("contents"));
    }
    catch (Exception err) {
      logger.debug("Exception saving: " + err.toString());
      if (err instanceof java.io.InvalidClassException) {
        try {
          ((FileChangeLog) WikiBase.getInstance().getChangeLogInstance()).deleteChangeTableFile(virtualWiki);
        }
        catch (Exception ignore) {
        }
        request.setAttribute("exception", new WikiServletException("Due to a change to the RecentChanges mechanism, " +
                                                                   "this upgrade is incompatible with previous recent changes, the offending file (wikihome/recent.hashtable) should now have been deleted, " +
                                                                   "please restart the app server"));
      }
      else
        request.setAttribute("exception", new WikiServletException(err.toString()));
      err.printStackTrace();
      RequestDispatcher dispatch = request.getRequestDispatcher("/jsp/servlet-error.jsp");
      dispatch.forward(request, response);
    }


    try {
      base.unlockTopic(virtualWiki, topic);
    }
    catch (Exception err) {
      request.setAttribute("exception", new WikiServletException(err.toString()));
      err.printStackTrace();
      RequestDispatcher dispatch = request.getRequestDispatcher("/jsp/servlet-error.jsp");
      dispatch.forward(request, response);
      return;
    }

    String next = null;
    response.setLocale(request.getLocale());
    next = JSPUtils.createRedirectURL(request, "Wiki?" + JSPUtils.encodeURL(topic,responseCharacterEncoding));
    logger.debug("Creating redirect: " + next);
    logger.debug("Redirect URL: " + response.encodeRedirectURL(next));
    response.sendRedirect(response.encodeRedirectURL(next));
  }

  protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
    response.sendError(HttpServletResponse.SC_BAD_REQUEST, "GET requests are not serviced in saves");
  }
}
