package vqwiki.lex;


/*
Very Quick Wiki - WikiWikiWeb clone
Copyright (C) 2001-2003 Gareth Cronin

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
GNU General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program (gpl.txt); if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA	02111-1307, USA.
*/

import java.io.*;
import java.net.*;
import java.util.*;
import vqwiki.Environment;
import vqwiki.WikiBase;
import vqwiki.PseudoTopicHandler;
import vqwiki.utils.Utilities;
import vqwiki.utils.JSPUtils;

%%

%public
%type String
%unicode
%extends	vqwiki.lex.Lexer
%class LinkLex

%init{
	yybegin( NORMAL );
%init}

%eofval{
	return null;
%eofval}

%{

	private static final String ISECSTART = "{${$${$$${--{";
	private static final String ISECEND = "}--}$$$}$$}$}";
	
	private static final String ATTACH = "attach:";
	private static final String REDIRECT = "redirect:";
	private static final String INCLUDE = "include:";
	private static final String MAILTO = "mailto:";
	private static final String IMAGEATTACH = ".gif.jpg.png.jpeg.bmp";
	private static final String DOCSPATH = "docs/";
	private static final String GRAPHSPATH = "graphics/";
	
	protected boolean isImage (String text) {
		String extension = null;
        int lastDot=text.lastIndexOf(".");
        if (lastDot >= 0) {
			extension = text.substring(lastDot).toLowerCase();
        } else {
        	return false;
        }
        if (IMAGEATTACH.indexOf (extension) >= 0) {
    	   	return true;
        }
        return false;
	}

    private boolean hasQuotes (String txt) {
    	int first = txt.indexOf("\"");
    	int last = txt.lastIndexOf("\"");
    	return (first >= 0) & (last >= 0) & (first != last);
    }
    
    private String getQuotedString (String txt) {
    	int first = txt.indexOf("\"");
   		int last = txt.lastIndexOf("\"");
    	if ((first >= 0) & (last >= 0) & (first != last)) {
    		return txt.substring (first + 1, last);
    	} else {
    		return txt;
    	}
    }
	protected String createAttachment(String yyText, String substitution) {
		String displayLink = yyText;

		String attachmentName = null;
		if (hasQuotes(displayLink)) {
			attachmentName = getQuotedString(displayLink);
		} else {
			if (displayLink.startsWith(ATTACH)) {
				attachmentName = displayLink.substring(ATTACH.length());
			} else {
				attachmentName = displayLink;
			}
		}
		String link = null;
        if (environment().uploadExists(virtualWiki, attachmentName)) {
            addDoc(attachmentName);
        	if (print & !preview) {
				if (attachmentName.indexOf (" ")>=0) {
					attachmentName = "\"" + attachmentName + "\"";
				}
				if (substitution != null) {
					return "<a>" + substitution + " (" + attachmentName + ")</a>";
				} else {
					return "<a>" + attachmentName + "</a>";
				}
			} else {
				if (export & !preview) {
				    String docPath = null;
				    if (attachmentName.lastIndexOf(".") < 0 ) {
		              docPath = DOCSPATH;                   		  
				    } else if (IMAGEATTACH.indexOf(attachmentName.substring(attachmentName.lastIndexOf(".")).toLowerCase()) < 0) {
		              docPath = DOCSPATH;                   		  
	                } else {
	                  docPath = GRAPHSPATH;
	                }				
					link = docPath + Utilities.encodeSafeFileName (attachmentName);
				} else {
					link = "Wiki?action=action_view_attachment&attachment=" + JSPUtils.encodeURL(attachmentName, charset);
				}
				StringBuffer buffer = new StringBuffer();
				buffer.append( "<a" );
				if (Environment.getInstance().getStringSetting(Environment.PROPERTY_ATTACHMENT_TYPE).equals("inline"))
					buffer.append(" target=\"_blank\"");
				buffer.append(" href=\"");
				buffer.append(link);
				buffer.append( "\" >" );
				if (substitution != null) {
					buffer.append(substitution);
				} else {
					if (attachmentName.indexOf(" ")>=0) {
						buffer.append("\"");
						buffer.append(attachmentName);
						buffer.append("\"");
					} else {
						buffer.append(attachmentName);
					}
				}
				buffer.append("</a>");
				return buffer.toString();
			}
		} else {
			if (substitution != null) {
				return substitution;
			} else {
				if (attachmentName.indexOf (" ")>=0) {
					attachmentName = "\"" + attachmentName + "\"";
				}
				return attachmentName;
			}
		}
	}

	protected String createImageAttachment(String yyText) {
		return createImageAttachment (yyText, null);
	}
	
	protected void handleRedirect(String yyText) {
		String topic = null;
		if (hasQuotes(yyText)) {
			topic = getQuotedString(yyText);
		} else {
			if (yyText.startsWith(REDIRECT)) {
				topic = yyText.substring(REDIRECT.length());
			} else {
				topic = yyText;
			}
		}
		addLink (topic);
		
	}
	
	protected String createImageAttachment(String yyText, String substitution) {
		String displayLink = yyText;

		String attachmentName = null;
		if (hasQuotes(displayLink)) {
			attachmentName = getQuotedString(displayLink);
		} else {
			if (displayLink.startsWith(ATTACH)) {
				attachmentName = displayLink.substring(ATTACH.length());
			} else {
				attachmentName = displayLink;
			}
		}

		StringBuffer sb = new StringBuffer ();
		if (substitution != null) {
		}
		if (environment().uploadExists(virtualWiki, attachmentName)) {
		    addDoc (attachmentName);
			if (export & !preview) {
			    String docPath = null;
                if (IMAGEATTACH.indexOf(attachmentName.substring(attachmentName.lastIndexOf(".")).toLowerCase()) < 0) {
	              docPath = DOCSPATH;                   		  
                } else {
                  docPath = GRAPHSPATH;
                }				
				sb.append("<img src=\"" + docPath + Utilities.encodeSafeFileName(attachmentName) + "\" alt=\"\" border=\"0\"/>");
			} else {
				String link = "Wiki?action=action_view_attachment&attachment=" + JSPUtils.encodeURL(attachmentName, charset);
				sb.append("<img src=\"" + link + "\" alt=\"\" border=\"0\"/>");
			}
		} else {
			sb.append(attachmentName);
		}
		if (substitution != null) {
			sb.append("</a>");
		}
		return sb.toString();
	}
	
	protected String createInclude(String yyText, String substitution) {
		if (hasQuotes(yyText)) {
			yyText = getQuotedString(yyText);
		} else {
			if (yyText.startsWith(INCLUDE)) {
				yyText = yyText.substring(INCLUDE.length());
			}
		}
		String contents = "";
		if (isImage(yyText)) {
			// is image
			String isImg = yyText;
			if (substitution != null) {
				boolean isTopic = false;
				if (substitution.startsWith("[")) {
					isTopic = true;
					substitution = substitution.substring (1);
					if (substitution.endsWith("]")) {
						substitution = substitution.substring (0, substitution.length()-1);
					}
				}
				substitution = substitution.trim();
				URL url = null;
				boolean isURL = false;
				if (!isTopic) {
					try {
						url = new URL(substitution);
						isURL = true;
					} catch (MalformedURLException e) {}
				}
				if (!isTopic && isURL) { // found an URL
					return createHyperlink (url, substitution, createImageAttachment(isImg));
				} else if (substitution.startsWith(ATTACH)) { // found an Attachment
				    return createAttachment(substitution, createImageAttachment(isImg));
				} else if (substitution.startsWith(INCLUDE)) { // found an Attachment
				    return createAttachment(substitution.substring(INCLUDE.length()), createImageAttachment(isImg));
				} else if (substitution.indexOf(":") >=0) { // found an external-link
					try {
						return LinkExtender.generateLink(
						    virtualWiki,
							substitution.substring( 0, substitution.indexOf( ':' ) ),
							substitution.substring( substitution.indexOf( ':' ) + 1 ),
							substitution,
							createImageAttachment(isImg),
							displayModeFullInfo
						);
					} catch( Exception err ){
						logger.error( "error generating link from extender", err );
						return createLink(substitution, createImageAttachment(isImg));
					}
				} else {
					return createLink(substitution, createImageAttachment(isImg));
				}
			} else {
				return createImageAttachment(isImg);
			}
		} else if (yyText.toLowerCase().endsWith(".html") | yyText.toLowerCase().endsWith(".txt") | yyText.toLowerCase().endsWith(".htm") | yyText.toLowerCase ().endsWith(".text")) {
			// is html or Text
			String isHtmlOrText = yyText;
			StringBuffer sb = new StringBuffer();
			Environment en = Environment.getInstance();
			try {
				BufferedReader bin = new BufferedReader(new FileReader(en.uploadPath(virtualWiki, isHtmlOrText)));
				String line = null;
				while (bin.ready()){
					line = bin.readLine();
					sb.append(line);
					sb.append("\n");
				}
				contents = sb.toString();
				addDoc(isHtmlOrText);
				if (yyText.toLowerCase().endsWith(".html") | yyText.toLowerCase().endsWith(".htm")) {
					String lowContents = contents.toLowerCase();
					int htmlEnd = lowContents.indexOf("</html>");
					if (htmlEnd >= 0) {
						contents = contents.substring (0, htmlEnd);
					}
					int htmlBegin = lowContents.indexOf("<html>");
					if (htmlBegin >= 0) {
						contents = contents.substring(htmlBegin + 6);
					}
				} else {
					contents = "<pre class=\"includetext\">" + contents + "</pre>";
				}
			} catch (FileNotFoundException e) {
				logger.warn("Could not find file: " + en.uploadPath(virtualWiki, isHtmlOrText));
			} catch (IOException e) {
				logger.warn("IOException!", e);
			}
		} else {
			// is isTopic
			String isTopic = yyText.trim();
			if ((topic != null) && !topicsList.contains(topic) && exists(isTopic.trim())) { topicsList.add(topic); }
			if (topicsList.contains(isTopic) | ignoreWikiname(isTopic) | !exists(isTopic.trim()) | PseudoTopicHandler.getInstance().isPseudoTopic(isTopic)) { return ""; }
	    	addLink(isTopic);
			
			String linkString = "";
			if (export) {
				if (substitution != null) {
					linkString = substitution;
				} else {
					linkString = isTopic;
				}
			} else if (print) {
				if (substitution != null) {
					linkString = substitution;
				} else {
					linkString = isTopic;
				}
			} else {
				if (substitution != null) {
					linkString = "<a href=\"Wiki?" + JSPUtils.encodeURL (isTopic, charset) + "\">" + substitution + "</a>";
				} else {
					linkString = "<a href=\"Wiki?" + JSPUtils.encodeURL (isTopic, charset) + "\">" + isTopic + "</a>";
				}			
			}			
			
			topicsList.add(isTopic);
			List linkList = new ArrayList();
			List docList = new ArrayList();
			try {
				if ((substitution != null) && (substitution == "")) {
					contents = WikiBase.getInstance().readCooked(virtualWiki, isTopic, topicsList, linkList, docList, displayModeFullInfo);	
				} else {
					contents = "<span class=\"subHeader\">" + linkString + "</span><p/>" + WikiBase.getInstance().readCooked(virtualWiki, isTopic, topicsList, linkList, docList, displayModeFullInfo);
				}
				addLinks(linkList);
				addDocs (docList);
			} catch (Exception e) {} // ignore include on error
			if (print & !preview) {
				String link = JSPUtils.encodeURL(isTopic, charset);
		        link = Utilities.replaceString(link, "%", "$");
				contents = "<a name=\""+ link + "\"></a>" + contents;
			}
		}
		if (print) {
		    return "<div class=\"includePrint\">" + contents + "</div>";
		} else {
		    return "<div class=\"includeDisplay\">" + contents + "</div>";
		}
	}
	
	protected String createHyperlink (URL url, String link, String substitution) {
	    addUrl (link);
	    boolean mailto = false;
	    link = link.trim();
	    String display = link;
	    if (link.startsWith(MAILTO)) {
	    	mailto = true;
	    	display = link.substring(MAILTO.length());
	    }
		if (substitution != null) {
			if (print) {
				if (mailto) {
					return "<a href=\"" + url.toString() + "\">" + substitution + " (" + display + ")</a>";
				} else {
					return "<a href=\"" + url.toString() + "\" target=\"_blank\">" + substitution + " (" + display + ")</a>";
				}
			} else {
				if (mailto) {
					return "<a href=\"" + url.toString() + "\">" + substitution + "</a>";
				} else {
					return "<a href=\"" + url.toString() + "\" target=\"_blank\">" + substitution + "</a>";
				}
			}
		} else {
			if (mailto) {
				return "<a href=\"" + url.toString() + "\">" + display + "</a>";
			} else {
				return "<a href=\"" + url.toString() + "\" target=\"_blank\">" + display + "</a>";
			}
		}
	}
	

	protected String createLink (String link) {
		return createLink(link, null);
	}
	
	protected String createLink (String link, String substitution) {
		link = link.trim();
		String display = link;
		if (substitution != null) {
			display = substitution;
		}
		if (ignoreWikiname(link)) {
			return link;
		} else if (exists(link)){
			if (export & !preview & PseudoTopicHandler.getInstance().isPseudoTopic(link) & !link.equals("WikiSearch")) {
				return "<span class=\"brokenLink\">" + display + "</span>";
			}
		    addLink (link);
			if (export & !preview) {
				return "<a href=\"" + Utilities.encodeSafeFileName (link) + ".html\">" + display + "</a>";
			} else if (print & !preview) {
				return "<a href=\"#" + Utilities.replaceString (JSPUtils.encodeURL (link, charset), "%", "$") +"\">" + display + "</a>";
			} else {
				return "<a href=\"Wiki?" + JSPUtils.encodeURL (link, charset) + "\">" + display + "</a>";
				
			}
		} else if (Utilities.checkURLChars(link)) {
		    // addLink (link);
			if ((export | print) &!preview) {
				return "<span class=\"brokenLink\">" + display + "</span>";
			} else {
				return "<a href=\"Wiki?topic=" + JSPUtils.encodeURL(link, charset) + "&action=action_edit\" class=\"new\">" + display + "</a>";
			}
		} else {
			return display;
		}
	}	
%}

whitespace = ([\t\ \r\n])
notbacktick_tabcrlf = ([^`\t\r\n])
notbacktick_crlf = ([^`\t\r\n])
notsquares = ([^\]{2}])
extlinkchar = ([^\t\ \r\n\<\>])
wikiname = (([:uppercase:]+[:lowercase:]+)([:uppercase:]+[:lowercase:]+)+)
link2 = (`{notbacktick_tabcrlf}+`)
link3 = ((\[\[{notbacktick_crlf}+\|{notbacktick_crlf}+\]\])|(\[\[{notbacktick_crlf}+\]\]))
hyperlink = ((http|ftp|mailto|news|https|telnet|file):{extlinkchar}+)
imagehyperlink = ({hyperlink}(\.gif|\.jpg|\.png|\.jpeg|\.GIF|\.JPG|\.PNG|\.JPEG|\.bmp|\.BMP))
extlink = (([:letter:]|[:digit:])+:{extlinkchar}+)
noformat = (__)
externalstart = (\[<[^>]+>\])
externalend = (\[<\/[:letter:]+>\])
attachment = (attach:{extlinkchar}+)
attachment2 = (attach:\".+\")
include = (include:{extlinkchar}+)
include2 = (include:\".+\")
redirect = (redirect:{extlinkchar}+)
redirect2 = (redirect:\".+\")

nonisecchar = ([^\{\}])
intersect = (\{\$\{\$\$\{\$\$\$\{--\{{nonisecchar}+\}--\}\$\$\$\}\$\$\}\$\})
intersect2 = (\{\$\{\$\$\{\$\$\$\{--\{\}--\}\$\$\$\}\$\$\}\$\})


%state NORMAL, OFF, PRE, EXTERNAL

%%

<NORMAL>{intersect2} {
	// {$$--$$} found
	return "";
	
}

<NORMAL>{intersect}	{
	// {$$-...-$$} found
	boolean decodeSubsection = true;
	String decodeSubsectionStr = (String)getParameter("decodeSubsection");
	if (decodeSubsectionStr != null) {
		decodeSubsection = Boolean.parseBoolean(decodeSubsectionStr);
	}
	if (decodeSubsection) {
		String line = yytext();
		while ((line.indexOf(ISECSTART) >= 0) & (line.indexOf(ISECEND) >= 0)) {
			int posB = line.indexOf(ISECSTART);
			int posE = line.indexOf(ISECEND);
			String start = line.substring (0, posB);
			String end = line.substring (posE + ISECEND.length());
			String middle = line.substring (posB + ISECSTART.length(), posE);
			line = start + Utilities.decodeSafeFileName(middle) + end;
		}
		return line;
	} else {
		return yytext();
	}
	
}

<NORMAL>\\{noformat}	{

	logger.debug( "escaped double backslash" );
	return "__";

}

<NORMAL>{noformat}	{

	logger.debug( "off" );
	yybegin( OFF );

}

<OFF>{noformat}	{

	logger.debug( "on" );
	yybegin( NORMAL );

}

<NORMAL, PRE>{externalstart} {

	logger.debug( "external" );
	yybegin( EXTERNAL );

}

<EXTERNAL>{externalend} {

	logger.debug( "external end");
	yybegin( NORMAL );

}

<NORMAL>(<pre>) {

	logger.debug( "@@@@{newline} entering PRE" );
	yybegin( PRE );
	return yytext();

}

<PRE>(<\/pre>) {

	logger.debug( "{newline}x2 leaving pre" );
	yybegin( NORMAL );
	return yytext();

}

<NORMAL>{imagehyperlink} {

	//hyperlink to an image like http://www.spiegel.de/static/sys/logo_120x61.gif

	logger.debug( "{imagehyperlink}" );
	String link = yytext();
	addUrl (link);
	return "<img src=\"" + link.trim() + "\" alt=\"\" border=\"0\"/>";

}

<NORMAL>{hyperlink} {

	// hyperlink like http://www.spiegel.de/

	logger.debug( "{hyperlink}" );
	String link = yytext();
	String punctuation = Utilities.extractTrailingPunctuation(link);

	if (punctuation!=null){
		link = link.substring(0, link.length()-punctuation.length()).trim();
	}

	String display = link;
	boolean isMail = false;
	if (link.startsWith (MAILTO)) {
		display = display.substring(MAILTO.length());
		isMail = true;
	}
	int pos = link.indexOf("|");
		if (pos >= 0) {
		display = link.substring(pos+1);
		display = display.replace('+', ' ');
		display = Utilities.replaceString(display, "\\ ", "+");  
 		link = link.substring(0, pos).trim();
	}
	if (isMail) {
		return "<a href=\"" + link + "\">" + display + "</a>" + punctuation;
	} else {
		return "<a href=\"" + link + "\" target=\"_blank\">" + display + "</a>" + punctuation;
	}
}


<NORMAL>{link2} {

	// Link to a topic, surounded by `...`

	logger.debug( "{link2}" );
	if( !Environment.getInstance().isAllowBackTick() ){
		logger.debug( "No back-tick links allowed" );
	} else {
		String link = yytext().substring(1, yytext().length() - 1).trim();
		return createLink (link);
	}
	return "";
}

<NORMAL>{link3} {

	// Link to a topic, surounded by [[...]]

	logger.debug( "{link3}" );
	String link = yytext();
	link = link.substring(2, link.length() - 2);

	// this is a bad hack ! :-( 
	// [[..]] is to greedy and takes [[..]]..[[..]] too.
	// so we have to split the rest and run it through the Link-Lexer
	String rest = null;
	int twoBracks = link.indexOf("]]");
	if (twoBracks>=0) {
		rest = link.substring(twoBracks + 2) + "]]";
		link = link.substring (0, twoBracks);
	}
	if (rest != null) {
		LinkLex restLex = new LinkLex(new StringReader(rest));
		restLex.setVirtualWiki(this.virtualWiki);
		restLex.setDisplayMode(this.displayMode);
		StringBuffer contents = new StringBuffer();
		while (true) {
			String line = null;
			try {
				line = restLex.yylex();
			} catch (Throwable e) {
				logger.debug(e);
			}
			if (line == null) { break; }
			contents.append(line);
		}
		rest = contents.toString();
		addLinks (restLex.getLinks());
		addDocs (restLex.getDocs());
		addUrls (restLex.getUrls());
	}
	if (rest == null) {rest = "";}
	String substitution = null;
	int sep = link.indexOf("|");
	if (sep >=0) {
		substitution = link.substring(sep+1);
		link = link.substring(0, sep);
	}
	boolean isTopic = false;
	if (link.startsWith("[")) {
		isTopic = true;
		link = link.substring (1);
		if (link.endsWith("]")) {
			link = link.substring (0, link.length()-1);
		}
	}
	link = link.trim();
	URL url = null;
	boolean isURL = false;
	if (!isTopic) {
		try {
			url = new URL(link);
			isURL = true;
		} catch (MalformedURLException e) {}
	}
	if (!isTopic && isURL) { // found an URL
		return createHyperlink (url, link, substitution) + rest;
	} else if (!isTopic & link.startsWith(ATTACH)) { // found an attachment
		return createAttachment(link, substitution) + rest;
	} else if (!isTopic & link.startsWith(INCLUDE)) { // found an include
		return createInclude(link, substitution) + rest;
	} else if(exists(link) ){ // found a Wiki-Topic-Link
		return createLink (link, substitution) + rest;
	} else if (!isTopic & (link.indexOf(":") >=0)) { // found an external-link
		try {
			return LinkExtender.generateLink(
			    virtualWiki,
				link.substring( 0, link.indexOf( ':' ) ),
				link.substring( link.indexOf( ':' ) + 1 ),
				link,
				substitution,
				displayModeFullInfo
			) + rest;
		} catch( Exception err ){
			logger.error( "error generating link from extender", err );
			return link + rest;
		}
    } else { // a not specified topic
		// addLink(link);
		if (substitution != null) {
			if ((export | print)& !preview) {
				return "<span class=\"brokenLink\">" + substitution + "</span>" + rest;
			} else {
				if (Utilities.checkURLChars(link)) {
					return "<a href=\"Wiki?topic=" + JSPUtils.encodeURL (link.trim(), charset) + "&action=action_edit\" class=\"new\">" + substitution + "</a>" + rest;
				} else {
				 	return "<span class=\"brokenLink\">" + substitution + "</span>" + rest;
				}
			}
		} else {
			if ((export | print)& !preview) {
				return "<span class=\"brokenLink\">" + link + "</span>" + rest;
			} else {
				if (Utilities.checkURLChars(link)) {
					return "<a href=\"Wiki?topic=" + JSPUtils.encodeURL (link.trim(), charset) + "&action=action_edit\" class=\"new\">" + link + "</a>" + rest;
				} else {
					return "<span class=\"brokenLink\">" + link + "</span>" + rest;
				}
			}
		}
	}

}

<NORMAL>{attachment2} {

	// attachment of a file, e.g. attach:"doc 97.pdf"

	logger.debug( "{attachment2}" );
	return createAttachment (yytext(), null);

}

<NORMAL>{attachment} {

	// attachment of a file, e.g. attach:doc.pdf

	logger.debug( "{attachment}" );
	return createAttachment(yytext(), null);

}

<NORMAL>{include2} {

	// include of a topic or html-file surrounded by "..."

	logger.debug( "{include2}" );
	return createInclude (yytext(), null);

}

<NORMAL>{include} {

	// include of a topic or html-file surrounded by "..."

	logger.debug( "{include2}" );
	return createInclude (yytext(), null);

}


<NORMAL>{redirect2} {

	// redirect surrounded by "..."

	logger.debug( "{redirect2}" );
	handleRedirect(yytext());
	return yytext();

}

<NORMAL>{redirect} {

	logger.debug( "{redirect}" );
	handleRedirect(yytext());
	return yytext();

}

<NORMAL>{extlink} {

	// External link to special pages, e.g. rfc:2501

	logger.debug( "{extlink}" );
	String text = yytext();
	try {
		return LinkExtender.generateLink(
		    virtualWiki,
			text.substring( 0, text.indexOf( ':' )),
			text.substring( text.indexOf( ':' ) + 1),
			text, null, displayModeFullInfo
		);
	} catch (Exception err) {
		logger.error( "error generating link from extender", err);
		return text;
	}

}


<NORMAL, OFF, PRE, EXTERNAL>{whitespace} {

	return yytext();

}

<NORMAL, OFF, PRE, EXTERNAL>.	{

 	logger.debug( ". (" + yytext() + ")" );
	return yytext();

}
