/**
 * Controls generation of external links based on the linking.properties file at the bottom
 * of the classpath.
 *
 * The file controls how link syntax is hyperlinked
 * Syntax: prefix=mylink
 * mylink can include the keyword expansion ${url} to add the suffix of the link
 * if you want to have the external link open in a new frame, or in a different frame,
 * a key of the form "prefix.target" can be specified, e.g.:
 *
 * c2=http://c2.com/cgi/wiki?${url}
 * c2.target=#blank
 *
 * @author garethc
 * 8/11/2002 08:53:38
 */
package vqwiki.lex;

import org.apache.log4j.Logger;

import vqwiki.VersionManager;
import vqwiki.WikiBase;
import vqwiki.utils.Utilities;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.util.Date;
import java.util.Properties;
import java.util.StringTokenizer;
import java.util.Vector;

public class LinkExtender {

  private static final Logger logger = Logger.getLogger(LinkExtender.class);
  private static final String LINKING_PROPERTIES_FILE = "/linking.properties";
  private static final String LINKINGPROPERTIES = "LinkingProperties";
  public static final String URL_KEYWORD_START = "${url";
  public static final String URL_KEYWORD_END = "}";
  private static Properties linkingProperties;
  private static Date linkingPropertiesDate = null;
  private static String EXPORT_EXT = ".export";

  /**
   * Use the prefix to produce a hyperlink based on entries in the linking.properties file
   *
   * @param prefix the prefix
   * @param url    the parameter as given by the user
   * @param text
   * @param displayName	the text to display
   * @return
   * @throws Exception
   */
  public static String generateLink(String virtualWiki, String prefix, String url, String text, String displayName) throws Exception {
  	return generateLink(virtualWiki, prefix, url, text, displayName, WikiBase.DISPLAY);
  }
  /**
   * Use the prefix to produce a hyperlink based on entries in the linking.properties file
   *
   * @param prefix the prefix
   * @param url    the parameter as given by the user
   * @param text
   * @return
   * @throws Exception
   */
  public static String generateLink(String virtualWiki, String prefix, String url, String text) throws Exception {
  	return generateLink(virtualWiki, prefix, url, text, null, WikiBase.DISPLAY);
  }

  	/**
   * Use the prefix to produce a hyperlink based on entries in the linking.properties file
   *
   * @param prefix the prefix
   * @param url    the parameter as given by the user
   * @param text
   * @return
   * @throws Exception
   */
  public static String generateLink(String virtualWiki, String prefix, String url, String text, String displayName, int displayMode) throws Exception {
    logger.debug("generating link for prefix=" + prefix + ",url=" + url);

    WikiBase wb = null;
	try {
		wb = WikiBase.getInstance();
	} catch (Exception err) {
		logger.error(err);
	}
    
    boolean export = (displayMode == WikiBase.EXPORT) | (displayMode == WikiBase.EXPORTPREV);
    boolean print = (displayMode == WikiBase.PRINT) | (displayMode == WikiBase.PRINTPREV);
    boolean preview = (displayMode == WikiBase.PRINTPREV) | (displayMode == WikiBase.EXPORTPREV);

    if (displayName == null) {
	    int displayNameDelimLocation = url.indexOf('|');
	    if (displayNameDelimLocation >= 0) {
	      displayName = url.substring(displayNameDelimLocation + 1);
	      displayName = displayName.replace('+', ' ');
	      displayName = displayName.replaceAll("\\\\ ", "+");
	      url = url.substring(0, displayNameDelimLocation);      
	    }
    }
    String extension = "";
    if (export & !preview) { extension = EXPORT_EXT; }
    Properties linkingProperties = getLinkingProperties(virtualWiki);
    String expansion = linkingProperties.getProperty(prefix + extension);
    if (expansion == null) {
    	expansion = linkingProperties.getProperty(prefix);
    }
    if (expansion == null) {
      logger.info("no expansion found for link extension: " + prefix);
      if (displayName != null) {
      		return displayName;
      } else {
      	return text;
      }
    }
    
    StringTokenizer st = new StringTokenizer (url, ":");
    Vector urls = new Vector ();
    int urlCount = 1;
    while (st.hasMoreTokens()) {
    	String token = st.nextToken();
    	urls.add(token);    
    	if (prefix.startsWith("vq")){
    		if (export & !preview) {
	    		token = Utilities.encodeSafeFileName(token);    			
    		}
    	} else {
    		if (export & !preview) {
    			token = token.replace('+', ' ');
    			token = token.replaceAll("\\\\ ", "+");
    		} else {
    			token = token.replaceAll("\\\\\\+", "+");
    		}
    	}
    	String urlNbr = "";
    	if (urlCount!=1) { urlNbr = Integer.toString (urlCount); }
    	String urlKeyword = URL_KEYWORD_START + urlNbr + URL_KEYWORD_END;
	    int urlLocation = expansion.indexOf(urlKeyword);	      
	    while (urlLocation >= 0) {
	      StringBuffer buffer = new StringBuffer();
	      buffer.append(expansion.substring(0, urlLocation));
	      buffer.append(token);
	      buffer.append(expansion.substring(urlLocation + urlKeyword.length(), expansion.length()));
	      expansion = buffer.toString();
	      urlLocation = expansion.indexOf(urlKeyword);
	    }
	    urlCount++;
    }
    if (prefix.startsWith("vq")){
    	String vqWiki = (String)urls.elementAt(0);
    	String vqTopic = (String) urls.elementAt(1);
    	if (!wb.exists(vqWiki, vqTopic)) {
    		return text;
    	}
    }
    String target = linkingProperties.getProperty(prefix + ".target");
    StringBuffer buffer = new StringBuffer();
    buffer.append("<a href=\"");
    buffer.append(expansion);
    buffer.append("\" ");
    if (!print& (target != null)) {
      buffer.append("target=\"");
      buffer.append(target);
      buffer.append("\"");
    }
    buffer.append(">");
    String urlDisplay = url.replaceAll("\\\\\\+", "+");    		
	text = text.replaceAll("\\\\\\+", "+");    		
    if (displayName == null) {
      String showPrefix = linkingProperties.getProperty(prefix + ".showprefix");
      if (showPrefix != null && showPrefix.equals("false")) {
        buffer.append(urlDisplay);
      } else {
        buffer.append(text);
      }
    } else {
        buffer.append(displayName);
    }
	if (print) {
		if (prefix.startsWith("vq")) {
			if (displayName != null) {
				if (displayName.equals(urls.elementAt(1))) {
					buffer.append(" (@ ");
					buffer.append(urls.elementAt(0));
					buffer.append(")");
				} else {
					buffer.append(" (");
					buffer.append(urls.elementAt(1));
					buffer.append(" @ ");					
					buffer.append(urls.elementAt(0));
					buffer.append(")");				
				}
			}
		} else {
			buffer.append(" (");
			buffer.append(expansion);
			buffer.append(")");
		}			
	}
    buffer.append("</a>");
    return buffer.toString();
  }

  private static Properties getLinkingProperties(String virtualWiki) throws IOException {
  	WikiBase wb = null;
  	VersionManager vm = null;
  	Date latestLinkingPropertiesDate = null;
  	try {
  		wb = WikiBase.getInstance();
  		vm = wb.getVersionManagerInstance();
  	  	latestLinkingPropertiesDate = vm.lastRevisionDate(virtualWiki, LINKINGPROPERTIES);
  	} catch (Exception e) {}
    if ((linkingPropertiesDate == null) || (latestLinkingPropertiesDate == null) || (linkingPropertiesDate.before(latestLinkingPropertiesDate))) {
      linkingProperties = new Properties();
      linkingPropertiesDate = latestLinkingPropertiesDate;
      try {
      	String linkingPropertiesTxt = wb.readRaw(virtualWiki, LINKINGPROPERTIES);
      	StringReader sr = new StringReader (linkingPropertiesTxt);
      	InputStream is = new ByteArrayInputStream(linkingPropertiesTxt.getBytes());
      	linkingProperties.load (is);    	
      } catch (Exception e) {}

      //      linkingProperties.load(Environment.getInstance().getResourceAsStream(LINKING_PROPERTIES_FILE));
    }
    return linkingProperties;
  }

}
