/*
Very Quick Wiki - WikiWikiWeb clone
Copyright (C) 2001-2002 Gareth Cronin

This program is free software; you can redistribute it and/or modify
it under the terms of the latest version of the GNU Lesser General
Public License as published by the Free Software Foundation;

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program (gpl.txt); if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package vqwiki;

import org.apache.log4j.Logger;
import vqwiki.db.DBDate;
import vqwiki.db.DatabaseChangeLog;
import vqwiki.db.DatabaseHandler;
import vqwiki.db.DatabaseNotify;
import vqwiki.db.DatabaseSearchEngine;
import vqwiki.db.DatabaseVersionManager;
import vqwiki.db.DatabaseWikiMembers;
import vqwiki.file.FileChangeLog;
import vqwiki.file.FileExtensionFilter;
import vqwiki.file.FileHandler;
import vqwiki.file.FileNotify;
import vqwiki.file.FileSearchEngine;
import vqwiki.file.FileVersionManager;
import vqwiki.file.FileWikiMembers;
import vqwiki.lex.*;
import vqwiki.lex.LexExtender;
import vqwiki.lex.Lexer;
import vqwiki.utils.Utilities;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeSet;


/**
 * Base class for Wiki. Handles all basic Viki stuff.
 *
 * @author $author$
 * @version $Revision: 1.58 $
 */
public class WikiBase {

  /**
   * An instance to myself. Singleton pattern.
   */
  private static WikiBase instance;

  /**
   * The topics are stored in a flat file
   */
  public static final int FILE = 0;

  /**
   * The topics are stored in a database
   */
  public static final int DATABASE = 1;

  /**
   * Name of the default wiki
   */
  public static final String DEFAULT_VWIKI = "jsp";

  /**
   * Name of the Plugins-Directory
   */
  public static final String PLUGINS_DIR = "plugins";
  
  // depending on the circumstances, different things should happen
  public static final int DISPLAY = 0;
  public static final int EXPORT = 1;
  public static final int PRINT = 2;
  public static final int EXPORTPREV = 3;
  public static final int PRINTPREV = 4;
  public static final int DEFAULT = DISPLAY;

  /**
   * Log output
   */
  private static final Logger logger = Logger.getLogger(WikiBase.class);

  /**
   * The handler that looks after read/write operations for a persitence type
   */
  protected Handler handler;

  /**
   * Listeners for topic changes
   */
  private List topicListeners;

  /**
   * Number of virtual wikis
   */
  private int virtualWikiCount;

  /**
   * Creates a new WikiBase object.
   *
   * @param persistenceType Which type of persistence is used
   * @throws Exception If the handler cannot be instanciated.
   */
  private WikiBase(int persistenceType)
  throws Exception {
    switch (persistenceType) {
      case FILE:
        this.handler = new FileHandler();
        break;

      case DATABASE:
        this.handler = new DatabaseHandler();
        break;
    }

    new SearchRefreshThread(
        Environment.getInstance()
        .getIndexRefreshInterval()
    );

    PluginManager.getInstance().installAll();
    this.topicListeners = new ArrayList();
    this.topicListeners.addAll(PluginManager.getInstance().getTopicListeners());
  }

  /**
   * Get a reference of this class. Implements singleton pattern.
   *
   * @return Instance of this class
   * @throws Exception If the storage produces errors
   */
  public static WikiBase getInstance()
  throws Exception {
    int persistenceType = Environment.getInstance().getPersistenceType();

    if (instance == null) {
      instance = new WikiBase(persistenceType);
    }

    return instance;
  }

  /**
   * Get an instance to the search enginge.
   *
   * @return Reference to the SearchEngine
   * @throws Exception the current search engine
   */
  public SearchEngine getSearchEngineInstance()
      throws Exception {
    switch (Environment.getInstance().getPersistenceType()) {
      case FILE:
        return FileSearchEngine.getInstance();

      case DATABASE:
        return DatabaseSearchEngine.getInstance();

      default:
        return FileSearchEngine.getInstance();
    }
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public VersionManager getVersionManagerInstance()
      throws Exception {
    switch (Environment.getInstance().getPersistenceType()) {
      case FILE:
        return FileVersionManager.getInstance();

      case DATABASE:
        return DatabaseVersionManager.getInstance();

      default:
        return FileVersionManager.getInstance();
    }
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public ChangeLog getChangeLogInstance()
      throws Exception {
    switch (Environment.getInstance().getPersistenceType()) {
      case FILE:
        return FileChangeLog.getInstance();

      case DATABASE:
        return DatabaseChangeLog.getInstance();

      default:
        return FileChangeLog.getInstance();
    }
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param topic       TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public Notify getNotifyInstance(String virtualWiki, String topic)
      throws Exception {
    switch (Environment.getInstance().getPersistenceType()) {
      case FILE:
        return new FileNotify(virtualWiki, topic);

      case DATABASE:
        return new DatabaseNotify(virtualWiki, topic);

      default:
        return new FileNotify();
    }
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public WikiMembers getWikiMembersInstance(String virtualWiki)
      throws Exception {
    switch (Environment.getInstance().getPersistenceType()) {
      case FILE:
        return new FileWikiMembers(virtualWiki);

      case DATABASE:
        return new DatabaseWikiMembers(virtualWiki);

      default:
        return new FileWikiMembers(virtualWiki);
    }
  }

  /**
   * Register a topic listener
   *
   * @param listener listener
   */
  public void addTopicListener(TopicListener listener) {
    if (!this.topicListeners.contains(listener)) {
      this.topicListeners.add(listener);
    }
  }

  /**
   * Finds a default topic file and returns the contents
   */
  public static String readDefaultTopic(String topicName)
      throws Exception {
    String resourceName = "/" + topicName + ".txt";
    java.net.URL resource = WikiBase.class.getResource(resourceName);
    if (resource == null) {
      throw new IllegalArgumentException("unknown default topic: " + topicName);
    }

    File f = new File(WikiBase.class.getResource(resourceName).getFile());
    logger.debug("Found the default topic: " + f);

    // Previous implementation using Readers (UTF-8) was adding a \n to the end
    // of the file resulting in an unwanted <br> in pages, and causing problems
    // when rendering them in a layout of composed wiki pages
    // (top-area, bottom-area, etc).
    InputStream in = WikiBase.class.getResourceAsStream(resourceName);
    BufferedInputStream is = new BufferedInputStream(in);
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    BufferedOutputStream os = new BufferedOutputStream(baos);
    String output = null;
    try {
      int c;
      while ((c = is.read()) != -1) {
        os.write(c); // also... it uses the buffers
      }
      os.flush();
      output = baos.toString();
    }
    catch (IOException e) {
      e.printStackTrace();
    }
    finally {
      // Closes the streams, if necessary
      if (is != null) {
        try {
          is.close();
        }
        catch (Exception ignore) {
          logger.warn(
              "exception closing file (you can ignore this warning)",
              ignore
          );
        }
      }
      if (os != null) {
        try {
          os.close();
        }
        catch (Exception ignore) {
          logger.warn(
              "exception closing output stream (you can ignore this warning)",
              ignore
          );
        }
      }
      in.close();
    }
    return output;
  }

  /**
   * Reads a file and returns the raw contents. Used for the editing version.
   */
  public synchronized String readRaw(String virtualWiki, String topicName)
      throws Exception {
    return handler.read(virtualWiki, topicName);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param topicName   TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public synchronized boolean exists(String virtualWiki, String topicName)
      throws Exception {
  	if (topicName == null) return false;
  	
    if (PseudoTopicHandler.getInstance().isPseudoTopic(topicName)) {
      return true;
    }

    return handler.exists(virtualWiki, topicName);
  }

  /**
   * Returns a topic's content that has been formatted to Wiki conventions.
   *
   * @param virtualWiki the virtual wiki to use
   * @param topicName   the topic
   */
  public synchronized String readCooked(String virtualWiki, String topicName, List topicsList, List linkList, List docList, int displayMode)
      throws Exception {
    String s = handler.read(virtualWiki, topicName);
   	s = s + "\n\n";
    BufferedReader in = new BufferedReader(new StringReader(s));

	if (topicsList == null) {
		topicsList = new ArrayList();
	}

    String result = cook(in, virtualWiki, topicName, topicsList, linkList, docList, displayMode);
    // remove all ending \n or <br/>
    boolean found = true;
    while (found) {
    	if (result.endsWith("\n")) {
        	result = result.substring(0, result.length()-1);
    	} else if (result.endsWith("<br/>")) { 
        	result = result.substring(0, result.length()-5);
    	} else {
    		found = false;
    	}
    }
    return result;	 
  }

  /**
   * Returns a topic's content that has been formatted to Wiki conventions.
   *
   * @param virtualWiki the virtual wiki to use
   * @param topicName   the topic
   */
  public synchronized String readCooked(String virtualWiki, String topicName, int displayMode)
      throws Exception {
  	return readCooked (virtualWiki, topicName, null, new ArrayList(), new ArrayList(), displayMode);
  }
  
  /**
   * Returns a topic's content that has been formatted to Wiki conventions.
   *
   * @param virtualWiki the virtual wiki to use
   * @param topicName   the topic
   */
  public synchronized String readCooked(String virtualWiki, String topicName)
      throws Exception {
  	return readCooked (virtualWiki, topicName, WikiBase.DISPLAY);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki    TODO: DOCUMENT ME!
   * @param topicName      TODO: DOCUMENT ME!
   * @param formatLexClass TODO: DOCUMENT ME!
   * @param layoutLexClass TODO: DOCUMENT ME!
   * @param linkLexClass   TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  private synchronized String readCooked(String virtualWiki, String topicName,
                                        String formatLexClass, String layoutLexClass, String linkLexClass)
      throws Exception {
  	return readCooked(virtualWiki, topicName, formatLexClass, layoutLexClass, linkLexClass, WikiBase.DISPLAY);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki    TODO: DOCUMENT ME!
   * @param topicName      TODO: DOCUMENT ME!
   * @param formatLexClass TODO: DOCUMENT ME!
   * @param layoutLexClass TODO: DOCUMENT ME!
   * @param linkLexClass   TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public synchronized String readCooked(String virtualWiki, String topicName,
                                        String formatLexClass, String layoutLexClass, String linkLexClass, int displayMode)
      throws Exception {
  	return readCooked(virtualWiki, topicName, null, formatLexClass, layoutLexClass, linkLexClass, displayMode, null);
  }
  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki    TODO: DOCUMENT ME!
   * @param topicName      TODO: DOCUMENT ME!
   * @param formatLexClass TODO: DOCUMENT ME!
   * @param layoutLexClass TODO: DOCUMENT ME!
   * @param linkLexClass   TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public synchronized String readCooked(String virtualWiki, String topicName,
                                        String formatLexClass, String layoutLexClass, String linkLexClass, int displayMode, Map templateParams)
      throws Exception {
  	return readCooked(virtualWiki, topicName, null, formatLexClass, layoutLexClass, linkLexClass, displayMode, templateParams);
  }

  private synchronized String readCooked(String virtualWiki, String topicName, List topicsList,
        String formatLexClass, String layoutLexClass, String linkLexClass, int displayMode, Map templateParams)
  		throws Exception {

  	String s = handler.read(virtualWiki, topicName);
	s = s + "\n\n";
	BufferedReader in = new BufferedReader(new StringReader(s));

	if (topicsList == null) {
		topicsList = new ArrayList();
	}

    String result = cook(
			in, virtualWiki, topicName, topicsList, null, null,
			formatLexClass, layoutLexClass, linkLexClass, displayMode, null, templateParams
	);

    // remove all ending \n or <br/>
    boolean found = true;
    while (found) {
    	if (result.endsWith("\n")) {
        	result = result.substring(0, result.length()-1);
    	} else if (result.endsWith("<br/>")) { 
        	result = result.substring(0, result.length()-5);
    	} else {
    		found = false;
    	}
    }
    return result;	 
}

  /**
   * TODO: DOCUMENT ME!
   *
   * @param in          TODO: DOCUMENT ME!
   * @param virtualWiki TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws ClassNotFoundException    TODO: DOCUMENT ME!
   * @throws NoSuchMethodException     TODO: DOCUMENT ME!
   * @throws InstantiationException    TODO: DOCUMENT ME!
   * @throws IllegalAccessException    TODO: DOCUMENT ME!
   * @throws InvocationTargetException TODO: DOCUMENT ME!
   * @throws IOException               TODO: DOCUMENT ME!
   */
  public synchronized String cook(BufferedReader in, String virtualWiki, String topicName)
	  throws ClassNotFoundException, NoSuchMethodException,
	  InstantiationException, IllegalAccessException,
	  InvocationTargetException, IOException {
  	return cook (in, virtualWiki, topicName, null, WikiBase.DISPLAY);
  }
  
  /**
   * TODO: DOCUMENT ME!
   *
   * @param in          TODO: DOCUMENT ME!
   * @param virtualWiki TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws ClassNotFoundException    TODO: DOCUMENT ME!
   * @throws NoSuchMethodException     TODO: DOCUMENT ME!
   * @throws InstantiationException    TODO: DOCUMENT ME!
   * @throws IllegalAccessException    TODO: DOCUMENT ME!
   * @throws InvocationTargetException TODO: DOCUMENT ME!
   * @throws IOException               TODO: DOCUMENT ME!
   */
  private synchronized String cook(BufferedReader in, String virtualWiki, String topicName, List topicsList)
	  throws ClassNotFoundException, NoSuchMethodException,
	  InstantiationException, IllegalAccessException,
	  InvocationTargetException, IOException {
  	return cook (in, virtualWiki, topicName, topicsList, null, null, WikiBase.DISPLAY);
  }
  /**
   * TODO: DOCUMENT ME!
   *
   * @param in          TODO: DOCUMENT ME!
   * @param virtualWiki TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws ClassNotFoundException    TODO: DOCUMENT ME!
   * @throws NoSuchMethodException     TODO: DOCUMENT ME!
   * @throws InstantiationException    TODO: DOCUMENT ME!
   * @throws IllegalAccessException    TODO: DOCUMENT ME!
   * @throws InvocationTargetException TODO: DOCUMENT ME!
   * @throws IOException               TODO: DOCUMENT ME!
   */
  private synchronized String cook(BufferedReader in, String virtualWiki, String topicName, List topicsList,  int displayMode )
      throws ClassNotFoundException, NoSuchMethodException,
      InstantiationException, IllegalAccessException,
      InvocationTargetException, IOException {

    return cook(in, virtualWiki, topicName, topicsList, null, null, displayMode);
  }
  /**
   * TODO: DOCUMENT ME!
   *
   * @param in          TODO: DOCUMENT ME!
   * @param virtualWiki TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws ClassNotFoundException    TODO: DOCUMENT ME!
   * @throws NoSuchMethodException     TODO: DOCUMENT ME!
   * @throws InstantiationException    TODO: DOCUMENT ME!
   * @throws IllegalAccessException    TODO: DOCUMENT ME!
   * @throws InvocationTargetException TODO: DOCUMENT ME!
   * @throws IOException               TODO: DOCUMENT ME!
   */
  public synchronized String cook(BufferedReader in, String virtualWiki, String topicName, List topicsList, List linkList, List docList, int displayMode )
      throws ClassNotFoundException, NoSuchMethodException,
      InstantiationException, IllegalAccessException,
      InvocationTargetException, IOException {
    Environment en = Environment.getInstance();

    return cook(
        in, virtualWiki, topicName, topicsList, linkList, docList,
        en.getStringSetting(Environment.PROPERTY_FORMAT_LEXER),
        en.getStringSetting(Environment.PROPERTY_LAYOUT_LEXER),
        en.getStringSetting(Environment.PROPERTY_LINK_LEXER),
        displayMode
    );
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param in          TODO: DOCUMENT ME!
   * @param virtualWiki TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws ClassNotFoundException    TODO: DOCUMENT ME!
   * @throws NoSuchMethodException     TODO: DOCUMENT ME!
   * @throws InstantiationException    TODO: DOCUMENT ME!
   * @throws IllegalAccessException    TODO: DOCUMENT ME!
   * @throws InvocationTargetException TODO: DOCUMENT ME!
   * @throws IOException               TODO: DOCUMENT ME!
   */
  public synchronized String cook(BufferedReader in, String virtualWiki, String topicName, List topicsList, List linkList, List docList, int displayMode, Map parameters)
      throws ClassNotFoundException, NoSuchMethodException,
      InstantiationException, IllegalAccessException,
      InvocationTargetException, IOException {
    Environment en = Environment.getInstance();

    return cook(
        in, virtualWiki, topicName, topicsList, linkList, docList,
        en.getStringSetting(Environment.PROPERTY_FORMAT_LEXER),
        en.getStringSetting(Environment.PROPERTY_LAYOUT_LEXER),
        en.getStringSetting(Environment.PROPERTY_LINK_LEXER),
        displayMode, parameters, null
    );
  }
  
  /**
   * TODO: DOCUMENT ME!
   *
   * @param in             TODO: DOCUMENT ME!
   * @param virtualWiki    TODO: DOCUMENT ME!
   * @param formatLexClass TODO: DOCUMENT ME!
   * @param layoutLexClass TODO: DOCUMENT ME!
   * @param linkLexClass   TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws ClassNotFoundException    TODO: DOCUMENT ME!
   * @throws NoSuchMethodException     TODO: DOCUMENT ME!
   * @throws InstantiationException    TODO: DOCUMENT ME!
   * @throws IllegalAccessException    TODO: DOCUMENT ME!
   * @throws InvocationTargetException TODO: DOCUMENT ME!
   * @throws IOException               TODO: DOCUMENT ME!
   */
  private synchronized String cook(BufferedReader in, String virtualWiki, String topicName, List topicsList, 
                                  String formatLexClass, String layoutLexClass, String linkLexClass)
      throws ClassNotFoundException, NoSuchMethodException,
      InstantiationException, IllegalAccessException,
      InvocationTargetException, IOException {
  	return cook(in, virtualWiki, topicName, topicsList, null, null, formatLexClass, layoutLexClass, linkLexClass, WikiBase.DISPLAY);
  }
  /**
   * TODO: DOCUMENT ME!
   *
   * @param in             TODO: DOCUMENT ME!
   * @param virtualWiki    TODO: DOCUMENT ME!
   * @param formatLexClass TODO: DOCUMENT ME!
   * @param layoutLexClass TODO: DOCUMENT ME!
   * @param linkLexClass   TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws ClassNotFoundException    TODO: DOCUMENT ME!
   * @throws NoSuchMethodException     TODO: DOCUMENT ME!
   * @throws InstantiationException    TODO: DOCUMENT ME!
   * @throws IllegalAccessException    TODO: DOCUMENT ME!
   * @throws InvocationTargetException TODO: DOCUMENT ME!
   * @throws IOException               TODO: DOCUMENT ME!
   */
  private synchronized String cook(BufferedReader in, String virtualWiki, String topic, List topicsList, List linkList, List docList,
                                  String formatLexClass, String layoutLexClass, String linkLexClass, int displayMode)
      throws ClassNotFoundException, NoSuchMethodException,
      InstantiationException, IllegalAccessException,
      InvocationTargetException, IOException {
  	return cook(in, virtualWiki, topic, topicsList, linkList, docList, formatLexClass, layoutLexClass, linkLexClass, displayMode, null, null);
  }
  /**
   * TODO: DOCUMENT ME!
   *
   * @param in             TODO: DOCUMENT ME!
   * @param virtualWiki    TODO: DOCUMENT ME!
   * @param formatLexClass TODO: DOCUMENT ME!
   * @param layoutLexClass TODO: DOCUMENT ME!
   * @param linkLexClass   TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws ClassNotFoundException    TODO: DOCUMENT ME!
   * @throws NoSuchMethodException     TODO: DOCUMENT ME!
   * @throws InstantiationException    TODO: DOCUMENT ME!
   * @throws IllegalAccessException    TODO: DOCUMENT ME!
   * @throws InvocationTargetException TODO: DOCUMENT ME!
   * @throws IOException               TODO: DOCUMENT ME!
   */
  private synchronized String cook(BufferedReader in, String virtualWiki, String topic, List topicsList, List linkList, List docList,
                                  String formatLexClass, String layoutLexClass, String linkLexClass, int displayMode, Map parameters, Map templateParams)
      throws ClassNotFoundException, NoSuchMethodException,
      InstantiationException, IllegalAccessException,
      InvocationTargetException, IOException {
  	if (topicsList == null) {
  		topicsList = new ArrayList();
  	}
  	if (parameters == null) {
  		parameters = new HashMap();
  	}
  	if (templateParams == null) {
  		templateParams = new HashMap();
  	}
    StringBuffer contents = new StringBuffer();
    Lexer lexer;
    String formatted;
    logger.debug("Using format lexer: " + formatLexClass);

    Class clazz = Class.forName(formatLexClass);
    Constructor constructor = clazz.getConstructor(
        new Class[] {Reader.class}
    );
    lexer = (Lexer) constructor.newInstance(
        new Object[] {in}
    );
    lexer.setVirtualWiki(virtualWiki);
    lexer.setDisplayMode(displayMode);
    lexer.setTopicsList(topicsList);
    lexer.setTopic(topic);
    lexer.setAllParameters(parameters);

    boolean external = false;
    String tag = null;
    StringBuffer externalContents = null;

    while (true) {
      String line = null;
      try {
        line = lexer.yylex();
      }
      catch (Throwable e) {
        logger.debug(e);
      }
      // logger.debug(line);

      if (line == null) { break; }

      if (line.startsWith("[<")) {
        if (!external) {
          external = true;
          tag = line.substring(2, line.length() - 2);
          logger.debug("External lex call (tag=" + tag + ")");
          externalContents = new StringBuffer();
          contents.append(line);
        }
        else {
          external = false;

          HashMap extParameters = new HashMap ();
          tag = extractParameters(tag, extParameters);
          LexExtender le = LexExtender.getInstance ();
          String converted = "";
       	  ExternalLex extlexer = le.getLexerInstance(tag);
       	  if (extlexer == null) {
          	converted = "[&lt;?" + tag + "?&gt;]";    	  	
       	  } else {
	          try {        	
	          	extlexer.setVirtualWiki(virtualWiki);
	          	extlexer.setDisplayMode(displayMode);
	          	extlexer.setTopicsList(topicsList);
	          	extlexer.setTopic(topic);
	          	extlexer.setAllParameters(extParameters);
	          	extlexer.setAllTemplateParameters(templateParams);
	
	          	converted = extlexer.process(externalContents.toString());
	          	
	            addAll(linkList, extlexer.getLinks());
	            addAll(docList, extlexer.getDocs());
	            templateParams.putAll(extlexer.getTemplateParameters());
	                   	
	          } catch (Exception e) {
	          	converted = "[&lt;!" + tag + "!&gt;]";
	          }
       	  }
          
          if (converted != null) {
            contents.append(converted);
          }

          contents.append(line);
          logger.debug("External ends");
        }
      }
      else {
        if (!external) {
          contents.append(line);
        }
        else {
          externalContents.append(line);
        }
      }
    }

    formatted = contents.toString();


    addAll(linkList, lexer.getLinks());
    addAll(docList, lexer.getDocs());
    
    String laidOut = formatted;

    if (!(layoutLexClass.equals("null") || layoutLexClass.equals(""))) {
      logger.debug("Using layout lexer: " + layoutLexClass);
      contents = new StringBuffer();
      clazz = Class.forName(layoutLexClass);
      constructor = clazz.getConstructor(
          new Class[]{ Reader.class }
      );
      lexer = (Lexer) constructor.newInstance(
          new Object[] { new StringReader(formatted) }
      );
      lexer.setVirtualWiki(virtualWiki);
      lexer.setDisplayMode(displayMode);
      lexer.setTopicsList(topicsList);
      lexer.setTopic(topic);
      lexer.setAllParameters(parameters);

      while (true) {
      	String line = null;
      	try {
      		line = lexer.yylex();
      	} catch (Throwable e) {
      		logger.debug(e);
      	}

        if (line == null) { break; }
        contents.append(line);
      }
      
      addAll(linkList, lexer.getLinks());
      addAll(docList, lexer.getDocs());

      laidOut = contents.toString();
    }

    String linked = laidOut;

    if (!(linkLexClass.equals("null") || linkLexClass.equals(""))) {
      logger.debug("Using link lexer: " + linkLexClass);
      contents = new StringBuffer();
      clazz = Class.forName(linkLexClass);
      constructor = clazz.getConstructor(
          new Class[] { Reader.class }
      );
      lexer = (Lexer) constructor.newInstance(
          new Object[] { new StringReader(laidOut) }
      );
      lexer.setVirtualWiki(virtualWiki);
      lexer.setDisplayMode(displayMode);
      lexer.setTopicsList(topicsList);
      lexer.setTopic(topic);
      lexer.setAllParameters(parameters);

      while (true) {
      	String line = null;
      	try {
      		line = lexer.yylex();
      	} catch (Throwable e) {
      		logger.debug(e);      		
      	}

        if (line == null) { break; }
        contents.append(line);
      }

      addAll(linkList, lexer.getLinks());
      addAll(docList, lexer.getDocs());
      
      linked = contents.toString();
    }

    return linked;
  }

  private String extractParameters (String tag, HashMap parameter) {
	String tagName = null;
	StringTokenizer st = new StringTokenizer (tag," ");
	
	boolean complete = true;
	String varName = null;
	String value = null;
	String varValue = null;

	while (st.hasMoreElements()) {
		if (tagName == null) {
			tagName = st.nextToken();
		} else {
			if (!complete) {
				String cont = st.nextToken ();
				varValue = varValue + " " + cont;			
			} else {
				varValue = st.nextToken();
			}
		    int pos = varValue.indexOf("=");
		    if (pos > 0) {
		    	varName = varValue.substring(0, pos);
		    	value = varValue.substring (pos + 1);
		    	if (value.startsWith("'")) {
		    		if (value.endsWith("'")) {
		    			complete = true;
		    			value = value.substring (1, value.length()-1);
		    			parameter.put(varName.toLowerCase(), value);
		    		} else {
		    			complete = false;
		    		}
		    	} else if (value.startsWith("\"")) {
		    		if (value.endsWith("\"")) {
		    			complete = true;
		    			value = value.substring (1, value.length()-1);
		    			parameter.put(varName.toLowerCase(), value);
		    		} else {
		    			complete = false;
		    		}
		    	} else {
		    		parameter.put (varName.toLowerCase(), value);
		    		complete = false;
		    	}
		    }
		}
	}

	return tagName;
  }
  private static void addAll (List oldElements, List newElements) {
    if ((newElements!=null)&&(oldElements != null)) {
    	for (Iterator it = newElements.iterator(); it.hasNext();) {
    		String element = (String) it.next();
    		if (!oldElements.contains(element)) {
    			oldElements.add(element);
    		}
    	}
    }
  	
  }
  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param topicName   TODO: DOCUMENT ME!
   * @param contents    TODO: DOCUMENT ME!
   */
  private void fireTopicSaved(String virtualWiki, String topicName,
                              String contents, String user, Date time) {
    logger.debug("topic saved event: " + topicListeners);
    if (topicListeners == null) {
      return;
    }
    for (Iterator iterator = topicListeners.iterator(); iterator.hasNext();) {
      TopicListener listener = (TopicListener) iterator.next();
      listener.topicSaved(
          new TopicSavedEvent(
              virtualWiki, topicName,
              contents, user, time
          )
      );
    }
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param contents    TODO: DOCUMENT ME!
   * @param convertTabs TODO: DOCUMENT ME!
   * @param topicName   TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public synchronized void write(String virtualWiki, String contents,
                                 boolean convertTabs, String topicName,
                                 String user)
      throws Exception {
    fireTopicSaved(virtualWiki, topicName, contents, user, new Date());
    handler.write(virtualWiki, contents, convertTabs, topicName);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param topicName   TODO: DOCUMENT ME!
   * @param key         TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public synchronized boolean holdsLock(String virtualWiki, String topicName,
                                        String key)
      throws Exception {
    return handler.holdsLock(virtualWiki, topicName, key);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param topicName   TODO: DOCUMENT ME!
   * @param key         TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public synchronized boolean lockTopic(String virtualWiki, String topicName,
                                        String key)
      throws Exception {
    return handler.lockTopic(virtualWiki, topicName, key);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param topicName   TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public synchronized void unlockTopic(String virtualWiki, String topicName)
      throws Exception {
    handler.unlockTopic(virtualWiki, topicName);
  }

  /**
   * Get recent changes for a given number of days
   */
  public Collection listRecentChanges(String virtualWiki, int numDays)
      throws Exception {
    TreeSet changes = new TreeSet();

    try {
      ChangeLog changeLog = this.getChangeLogInstance();
      Calendar cal = Calendar.getInstance();

      for (int i = 0; i <= numDays; i++) {
        Date aDate = cal.getTime();
        Collection col = changeLog.getChanges(virtualWiki, aDate);

        if (col != null) {
          changes.addAll(col);
        }
      }

      cal.add(Calendar.DATE, -1);
    }
    catch (IOException e1) {
      logger.error(e1);
    }
    catch (ClassNotFoundException e1) {
      logger.error(e1);
    }

    return changes;
  }


  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param topicName   TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  protected boolean isTopicReadOnly(String virtualWiki, String topicName)
      throws Exception {
    return this.handler.isTopicReadOnly(virtualWiki, topicName);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @return TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public Collection getReadOnlyTopics(String virtualWiki)
      throws Exception {
    return this.handler.getReadOnlyTopics(virtualWiki);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param topicName   TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public void addReadOnlyTopic(String virtualWiki, String topicName)
      throws Exception {
    this.handler.addReadOnlyTopic(virtualWiki, topicName);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @param virtualWiki TODO: DOCUMENT ME!
   * @param topicName   TODO: DOCUMENT ME!
   * @throws Exception TODO: DOCUMENT ME!
   */
  public void removeReadOnlyTopic(String virtualWiki, String topicName)
      throws Exception {
    this.handler.removeReadOnlyTopic(virtualWiki, topicName);
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @return TODO: DOCUMENT ME!
   */
  protected String fileBase() {
    return Environment.getInstance().getHomeDir();
  }

  /**
   * TODO: DOCUMENT ME!
   *
   * @throws Exception TODO: DOCUMENT ME!
   */
  public static void initialise()
      throws Exception {
    int persistenceType = Environment.getInstance().getPersistenceType();
    WikiMail.init();
    instance = new WikiBase(persistenceType);
  }

  /**
   * Find all topics without links to them
   */
  public Collection getOrphanedTopics(String virtualWiki) throws Exception{
    Collection all = getSearchEngineInstance().getAllTopicNames(virtualWiki);  	
  	Collection display = getOrphanedTopics(virtualWiki, all, WikiBase.DISPLAY);
  	Collection export =  getOrphanedTopics(virtualWiki, display, WikiBase.EXPORT);
  	Collection print =  getOrphanedTopics(virtualWiki, export, WikiBase.PRINT);
  	return print;
  }

  /**
   * Find all topics without links to them
   */
  public Collection getOrphanedTopics(String virtualWiki, Collection all, int displayMode)
      throws Exception {
    WikiBase wb = WikiBase.getInstance();
    Collection results = new TreeSet();
    List excludeList = WikiBase.getAllSpecialPages();
    excludeList.add(Environment.getInstance().getDefaultTopic());
    excludeList.add(wb.getVirtualWikiHome(virtualWiki));
    
    for (Iterator iterator = all.iterator(); iterator.hasNext();) {
      String topicName = (String) iterator.next();
      Collection matches = getSearchEngineInstance().findLinkedTo(
          virtualWiki,
          topicName,
		  displayMode
      );
      logger.debug(topicName + ": " + matches.size() + " matches");
      if ((matches.size() == 0) && (!excludeList.contains(topicName))) {
        results.add(topicName);
      }
    }

    logger.debug(results.size() + " orphaned topics found");

    return results;
  }

  /**
   * Find all topics that haven't been written but are linked to
   */
  public Collection getToDoWikiTopics(String virtualWiki) throws Exception {
  	Collection display = getToDoWikiTopics(virtualWiki, WikiBase.DISPLAY);
  	Collection export =  getToDoWikiTopics(virtualWiki, WikiBase.EXPORT);
  	Collection print =  getToDoWikiTopics(virtualWiki, WikiBase.PRINT);
  	Collection result = new TreeSet();
  	result.addAll(display);
  	result.addAll(export);
  	result.addAll(print);
  	return result;
  }


  /**
   * Find all topics that haven't been written but are linked to
   */
  public Collection getToDoWikiTopics(String virtualWiki, int displayMode)
      throws Exception {
    Collection results = new TreeSet();
    Collection all = getSearchEngineInstance().getAllTopicNames(virtualWiki);

    Set topicNames = new HashSet();

    for (Iterator iterator = all.iterator(); iterator.hasNext();) {
      String topicName = (String) iterator.next();
      String content = handler.read(virtualWiki, topicName);
      StringReader reader = new StringReader(content);
      
      Environment en = Environment.getInstance();

      String linkLexClass = en.getStringSetting(Environment.PROPERTY_LINK_LEXER);
      Class clazz = Class.forName(linkLexClass);
      Constructor constructor = clazz.getConstructor(new Class[]{Reader.class});
      Lexer lexer = (Lexer) constructor.newInstance(new Object[]{reader});

      lexer.setVirtualWiki(virtualWiki);
      lexer.setDisplayMode(displayMode);
           
      while (true) {
      	String line = null;
      	try {
      		line = lexer.yylex();
      	} catch (Exception e) {
      		logger.debug(e);      		
      	}

        if (line == null) { break; }
      }
      
      reader.close();
      topicNames.addAll(lexer.getLinks());
    }

    for (Iterator iterator = topicNames.iterator(); iterator.hasNext();) {
      String topicName = (String) iterator.next();
      if (!PseudoTopicHandler.getInstance().isPseudoTopic(topicName)
          && !handler.exists(virtualWiki, topicName)
          && !"\\\\\\\\link\\\\\\\\".equals(topicName)) {
        results.add(topicName);
      }
    }

    logger.debug(results.size() + " todo topics found");

    return results;
  }

  /**
   * Return a list of all virtual wikis on the server
   */
  public Collection getVirtualWikiList()
      throws Exception {
    return this.handler.getVirtualWikiList();
  }

  /**
   * Return the homepage of a virtual wiki
   */

  public String getVirtualWikiHome(String virtualWiki) {
  	String defaultTopic = Environment.DEFAULT_DEFAULT_TOPIC;
  	try { 
  		defaultTopic = Utilities.resource("virtualwiki." + virtualWiki + ".home", Locale.getDefault());
    } catch (MissingResourceException e) {}
    return defaultTopic;
  }

  /**
   * Return the homepage of a virtual wiki
   */

  public String getVirtualWikiHomeTitle(String virtualWiki) {
  	String defaultTitle = getVirtualWikiHome(virtualWiki);
  	try { 
  		defaultTitle = Utilities.resource("virtualwiki." + virtualWiki + ".hometitle", Locale.getDefault());
    } catch (MissingResourceException e) {}
    return defaultTitle;
  }

  public void getEverythingFromStart (String virtualWiki, int displayMode, Collection reachedTopics, Collection reachedDocs) throws Exception {
    String TopArea = "TopArea";
    String LeftMenu = "LeftMenu";
    String BottomArea ="BottomArea";   
    String defaultTopic = getVirtualWikiHome (virtualWiki);
    String [] firstPage = {TopArea, LeftMenu, BottomArea, defaultTopic};
    List interest = new ArrayList ();
    for (int tc= 0; tc < firstPage.length; tc++) {
      interest.add(firstPage[tc]);
    }
    getEverythingFromHere (virtualWiki, interest, displayMode, reachedTopics, reachedDocs );
  }
  
  public void getEverythingFromHere (String virtualWiki, List startTopics, int displayMode, Collection reachedTopics, Collection reachedDocs) throws Exception{
  	int count = 0;
  	List topics = startTopics;
  	while (count < topics.size()){
  		String topicName = (String) topics.get(count);

  		reachedTopics.add(topicName);
        String content = handler.read(virtualWiki, topicName);
        StringReader reader = new StringReader(content);
        
        Environment en = Environment.getInstance();

        List topicsList = new ArrayList();
        List linkList = new ArrayList ();
        List docList = new ArrayList ();
		String contents = readCooked(virtualWiki, topicName, topicsList, linkList, docList, displayMode);

        for (Iterator it = linkList.iterator(); it.hasNext();) {
        	String newTopic = (String)it.next();
        	if (!topics.contains(newTopic)) {
        		topics.add(newTopic);
        	}
        }
        for (Iterator it = docList.iterator(); it.hasNext();) {
        	String newDoc = (String)it.next();
        	if (!reachedDocs.contains(newDoc)) {
        		reachedDocs.add(newDoc);
        	}
        }
  		count++;
  	}
  }
  /**
   * Calculate all visible pages, starting with the start-Page
   * excluding Top, Left, and Bottom
   * @param virtualWiki
   * @return List of all visible pages, starting with the start-Page
   * @throws Exception
   */
  public List getAllVisiblePages (String virtualWiki) throws Exception {
  	if ((virtualWiki == null) || (virtualWiki.equals(""))) {
  		virtualWiki = DEFAULT_VWIKI;
  	}
  	
    String TopArea = "TopArea";
    String LeftMenu = "LeftMenu";
    String BottomArea ="BottomArea";   
    String defaultTopic = getVirtualWikiHome (virtualWiki);

    SearchEngine sedb = getSearchEngineInstance();

    HashMap containingTopics = new HashMap();
    Collection all = sedb.getAllTopicNames(virtualWiki);
    Iterator allIterator = all.iterator();
    
    while (allIterator.hasNext()) {
          String topicname = (String) allIterator.next();
          Collection searchresult = sedb.find(virtualWiki, topicname, false);
          // Collection searchresult = null;
          if ((searchresult != null) && (searchresult.size() > 0)) {
               Iterator it = searchresult.iterator();
               for (; it.hasNext();) {
                     SearchResultEntry result = (SearchResultEntry) it.next();
                     if (!result.getTopic().equals(topicname)) {
                           List l = (List) containingTopics.get(result.getTopic());
                           if (l == null)
                                 l = new ArrayList();
                           if (!l.contains(topicname))
                                 l.add(topicname);
                           containingTopics.put(result.getTopic(), l);
                     }
                }

          }
    }
    String [] firstPage = {TopArea, LeftMenu, BottomArea, defaultTopic};
    List interest = new ArrayList ();
    for (int tc= 0; tc < firstPage.length; tc++) {
      interest.add(firstPage[tc]);
    }
    for (int tc= 0; tc < interest.size(); tc++) {
  	   List sons = (List) containingTopics.get(interest.get(tc));
  	   if (sons!= null) {
  	     for (int sc = 0; sc < sons.size(); sc++) {
  	 	    if (!interest.contains(sons.get(sc))) {
  	   		    interest.add(sons.get(sc));
    	 	  }
    	   }
  	   }
    }
    interest.remove(TopArea);
    interest.remove(LeftMenu);
    interest.remove(BottomArea);
    return interest;
  }
  /**
   * Return a list of available templates
   */
  public Collection getTemplates(String virtualWiki)
      throws Exception {
    return this.handler.getTemplateNames(virtualWiki);
  }

  /**
   * Return a given template
   */
  public String getTemplate(String virtualWiki, String name)
      throws Exception {
    return this.handler.getTemplate(virtualWiki, name);
  }

  /**
   * Purge deleted files
   *
   * @return a collection of strings that are the deleted topic names
   */
  public Collection purgeDeletes(String virtualWiki)
      throws Exception {
    return this.handler.purgeDeletes(virtualWiki);
  }

  /**
   * Add virtual wiki
   */
  public void addVirtualWiki(String virtualWiki)
      throws Exception {
    this.handler.addVirtualWiki(virtualWiki);
  }

  /**
   * purge versions older than a certain date
   */
  public void purgeVersionsOlderThan(String virtualWiki, DBDate date)
      throws Exception {
    this.handler.purgeVersionsOlderThan(virtualWiki, date);
  }

  /**
   * save the contents as a template
   */
  public void saveAsTemplate(String virtualWiki, String templateName,
                             String contents)
      throws Exception {
    this.handler.saveAsTemplate(virtualWiki, templateName, contents);
  }

  /**
   * get a count of the number of virtual wikis in the system
   *
   * @return the number of virtual wikis
   */
  public int getVirtualWikiCount() {
    if (virtualWikiCount == 0) {
      try {
        virtualWikiCount = getVirtualWikiList().size();
      }
      catch (Exception e) {
        logger.warn(e);
      }
    }

    return virtualWikiCount;
  }

  /**
   * return the current handler instance
   *
   * @return the current handler instance
   */
  public Handler getHandler() {
    return handler;
  }

  /**
   * Do emergency repairs by clearing all locks and deleting recent changes files
   */
  public void panic() {
    Collection wikis = null;
    try {
      wikis = getVirtualWikiList();
    }
    catch (Exception e) {
      logger.error("problem getting the virtual wiki list", e);
      return;
    }
    for (Iterator iterator = wikis.iterator(); iterator.hasNext();) {
      String virtualWikiName = (String) iterator.next();
      try {
        List lockList = handler.getLockList(virtualWikiName);
        for (Iterator lockIterator = lockList.iterator(); lockIterator.hasNext();) {
          TopicLock lock = (TopicLock) lockIterator.next();
          handler.unlockTopic(virtualWikiName, lock.getTopicName());
        }
      }
      catch (Exception e) {
        logger.error("", e);
      }
      // destroy recent changes
      if (Environment.getInstance().getPersistenceType() != DATABASE) {
        try {
          FileHandler.getPathFor(virtualWikiName, "recent.xml").delete();
        }
        catch (Exception e) {
          logger.error("error removing recent.xml", e);
        }
      }
      // failsafe
      if (Environment.getInstance().getPersistenceType() != DATABASE) {
        try {
          File wikiDir = FileHandler.getPathFor(virtualWikiName, null);
          File[] locks = wikiDir.listFiles(new FileExtensionFilter(".lock"));
          for (int i = 0; i < locks.length; i++) {
            File lock = locks[i];
            lock.delete();
          }
        }
        catch (Exception e) {
          logger.error("error removing recent.xml", e);
        }
      }
    }
  }

  /**
   * Return true if the given topic is marked as "admin only", i.e. it is present in the admin only topics topic
   *
   * @param virtualWiki
   * @param topicName
   * @return
   */
  public boolean isAdminOnlyTopic(Locale locale, String virtualWiki, String topicName) throws Exception {
    String adminOnlyTopics = readRaw(virtualWiki, Utilities.getMessages(locale).getString("specialpages.adminonlytopics"));
    StringTokenizer tokenizer = new StringTokenizer(adminOnlyTopics);
    while (tokenizer.hasMoreTokens()) {
      String token = tokenizer.nextToken();
      if(token.equals(topicName)){
        return true;
      }
    }
    return false;
  }
  
  public static List getAllSpecialPages () {
    ResourceBundle messages = Utilities.getMessages();
    StringTokenizer spTokens = new StringTokenizer(messages.getString("specialpages.specialpages"), ",");
    ArrayList list = new ArrayList ();
    while (spTokens.hasMoreElements()) {
    	String topic = null;
    	try {
    		topic = messages.getString("specialpages." + spTokens.nextToken());
    	} catch (Exception e) {}
    	if (topic != null) {
    		list.add(topic);
    	}
    }
  	return list;
  }

}
