/*
Very Quick Wiki - WikiWikiWeb clone
Copyright (C) 2001-2002 Gareth Cronin

This program is free software; you can redistribute it and/or modify
it under the terms of the latest version of the GNU Lesser General
Public License as published by the Free Software Foundation;

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program (gpl.txt); if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package vqwiki;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

/**
 * Class, which represents one topic
 *
 * @author $author$
 * @version $Revision: 1.15 $
 */
public class Topic {
  //~ Instance variables �����������������������������������������������������

  /** Reference to the wiki base class */
  protected WikiBase wb;

  /** Name of this topic */
  protected String topicName;

  /** Contents of this topic */
  protected String contents;

  /** Author of this topic */
  protected String author;

  /** Last modification date of this topic */
  protected Date lastRevisionDate;

  /** revision number of this topic */
  protected int revision;

  //~ Constructors �����������������������������������������������������������

  /**
   * Creates a new Topic object.
   *
   * @param topicName The name of this topic
   *
   * @throws Exception If the topic cannot be retrieved
   */
  public Topic(String topicName)
      throws Exception {
    this.topicName = topicName;
    wb = WikiBase.getInstance();
    this.contents = null;
    this.author = null;
    this.lastRevisionDate = null;
    this.revision = -1;
  }

  //~ Methods ����������������������������������������������������������������

  /**
   * Find the most recent revision before the current
   *
   * @param virtualWiki The virtualWiki, which contains the topic
   *
   * @return the last revision date, or null if versioning is off
   */
  public Date getMostRecentRevisionDate(String virtualWiki)
      throws Exception {
    if (Environment.getInstance().isVersioningOn()) {
      this.lastRevisionDate = wb.getVersionManagerInstance()
          .lastRevisionDate(virtualWiki, this.topicName);
      return this.lastRevisionDate;
    }
    else {
      return null;
    }
  }

  /**
   * Return a diff for the current vs the most recent revision before it.
   *
   * @param virtualWiki The virtualWiki, which contains the topic
   * @param useHtml if true, convert the line feeds to &lt;p&gt; tags
   *
   * @return a diff to the last revision
   */
  public String mostRecentDiff(String virtualWiki, boolean useHtml)
      throws Exception {
    String diff = wb.getVersionManagerInstance().diff(virtualWiki,
                                                   topicName, 0, 1);
    if(!useHtml){
      return diff;
    }
    StringBuffer b = new StringBuffer("<p>");

    for (int i = 0; i < diff.length(); i++) {
      if (diff.charAt(i) != '\n') {
        b.append(diff.charAt(i));
      }
      else {
        b.append("</p><p>");
      }
    }

    b.append("</p>");

    return b.toString();
  }

  /**
   * Get a diff for two arbitrary versions of a topic
   * @param virtualWiki the virtual wiki
   * @param firstVersion the first version number
   * @param secondVersion the second version number
   * @return
   */
  public String getDiff(String virtualWiki, int firstVersion, int secondVersion) throws Exception {
    String d = wb.getVersionManagerInstance().diff(virtualWiki,
                                                   topicName, firstVersion, secondVersion);
    StringBuffer b = new StringBuffer("<p>");

    for (int i = 0; i < d.length(); i++) {
      if (d.charAt(i) != '\n') {
        b.append(d.charAt(i));
      }
      else {
        b.append("</p><p>");
      }
    }

    b.append("</p>");

    return b.toString();

  }

  /**
   * Find the most recent author.
   *
   * @param virtualWiki The virtualWiki, which contains the topic
   *
   * @return the author, who editied the last revision, or null if versioning is off
   */
  public String getMostRecentAuthor(String virtualWiki)
      throws Exception {
    if (Environment.getInstance().isVersioningOn()) {
      // file last revision date:
      getMostRecentRevisionDate(virtualWiki);
      if (this.lastRevisionDate == null)
        return null;

      Collection c = WikiBase.getInstance().getChangeLogInstance().getChanges(virtualWiki, this.lastRevisionDate);
      if (c == null)
        return null;
      Iterator it = c.iterator();

      Date authorDate = null;

      while (it.hasNext()) {
        Change thischange = (Change) it.next();
        if (thischange.getTopic().equals(this.topicName)) {
          if (authorDate == null) {
            this.author = thischange.getUser();
            authorDate = thischange.getTime();
          }
          else {
            if (authorDate.before(thischange.getTime())) {
              this.author = thischange.getUser();
              authorDate = thischange.getTime();
            }
          }
        }
      }

      return this.author;
    }
    else {
      return null;
    }
  }

  /**
   * Find the revision number.
   *
   * @param virtualWiki The virtualWiki, which contains the topic
   *
   * @return the revision number
   */
  public int getRevision(String virtualWiki)
      throws Exception {
    this.revision = wb.getVersionManagerInstance().
        getNumberOfVersions(virtualWiki, this.topicName);
    return this.revision;
  }

  /**
   * Make a topic read-only
   *
   * @param virtualWiki The virtualWiki, which contains the topic
   */
  public synchronized void makeTopicReadOnly(String virtualWiki)
      throws Exception {
    wb.addReadOnlyTopic(virtualWiki, topicName);
  }

  /**
   * Return whether a topic is read-only
   *
   * @param virtualWiki The virtualWiki, which contains the topic
   */
  public boolean isReadOnlyTopic(String virtualWiki)
      throws Exception {
    return wb.isTopicReadOnly(virtualWiki, topicName);
  }

  /**
   * Make a previously read-only topic writable
   *
   * @param virtualWiki The virtualWiki, which contains the topic
   */
  public synchronized void makeTopicWritable(String virtualWiki)
      throws Exception {
    wb.removeReadOnlyTopic(virtualWiki, topicName);
  }
}
