/**
 * Copyright 2004, Gareth Cronin
 * User: garethc
 * Date: 6/06/2004
 * Time: 09:19:14
 */
package vqwiki;

import org.apache.log4j.Logger;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.util.Properties;

/**
 * Manager that maintains bindings between logical action names and classes that implement {@link WikiAction}
 * The action manager can delegate requests to appropriate actions.
 */
public class ActionManager {

  /**
   * Logger
   */
  public static final Logger logger = Logger.getLogger(ActionManager.class);

  /**
   * Mapping of action names to classes
   */
  private Properties mapping;

  /**
   * Singleton instance
   */
  private static ActionManager ourInstance = new ActionManager();

  /**
   * Name of the resource that is the mapping file to persist to
   */
  private static final String MAPPING_PROPERTIES_FILE = "/actions.properties";

  /**
   * Get singleton instance
   * 
   * @return singleton instance
   */
  public static ActionManager getInstance() {
    return ourInstance;
  }

  /**
   * Hide constructor
   */
  private ActionManager() {
  }

  /**
   * Return an instance of the named action
   *
   * @param actionName name
   * @return instance or null if there is no mapping for the named action
   * @throws ClassNotFoundException if the mapped class can't be found
   * @throws IllegalAccessException on a security problem
   * @throws InstantiationException if the mapped class can't be instantiated
   */
  public WikiAction getActionInstance(String actionName) throws ClassNotFoundException, IllegalAccessException,
      InstantiationException {
    logger.debug("getting action instance: " + actionName);
    String className = getMapping().getProperty(actionName);
    if (className == null) {
      return null;
    }
    Class clazz = Class.forName(className);
    return (WikiAction) clazz.newInstance();
  }

  /**
   * Get the current mapping
   *
   * @return mapping
   */
  private Properties getMapping() {
    if (mapping == null) {
      InputStream in = ActionManager.class.getResourceAsStream(MAPPING_PROPERTIES_FILE);
      if (in == null) {
        logger.warn(MAPPING_PROPERTIES_FILE + " not found");
      }
      mapping = new Properties();
      try {
        mapping.load(in);
        in.close();
      }
      catch (IOException e) {
        logger.warn(e);
      }
    }
    logger.debug("mapping: " + mapping);
    return mapping;
  }

  /**
   * Add a mapping
   *
   * @param actionName action name
   * @param className  class implementing {@link WikiAction}
   * @throws IOException on any error persisting the mapping
   */
  public void addMapping(String actionName, String className) throws IOException {
    logger.debug("adding action mapping: " + actionName + "->" + className);
    getMapping().setProperty(actionName, className);
    URL url = ActionManager.class.getResource(MAPPING_PROPERTIES_FILE);
    FileOutputStream out = null;
    try {
        File f = new File (new URI (url.toString()));
        out = new FileOutputStream (f);
        // out = new FileOutputStream(new File(base.getFile(), MAPPING_PROPERTIES_FILE));
     
      getMapping().store(out, "actions");
    } catch (Exception e) {
    	e.printStackTrace();
    } finally {
      if (out != null) {
        out.close();
      }
    }
  }

  /**
   * Return whether a mapping exists for the named action
   *
   * @param actionName action name
   * @return true if it exists
   */
  public boolean actionExists(String actionName) {
    boolean result = getMapping().containsKey(actionName);
    logger.debug("action exists: '" + actionName + "' ?" + result);
    return result;
  }
}
