/*
 * TemplateEngine for Java
 * Copyright (C) 2003 Niels Wojciech Tadeusz Andersen <haj@zhat.dk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Changes:
 * - 17.05.05, Uwe Roth
 *   added: private void init ()
 *   added: public Template(byte[] content) throws Exception
 *   added: public void setContent(byte[] content) throws Exception
 */

package TemplateEngine;
import java.io.*;

public class Template
{
	private final static String DELIMITER = "@@";
	private final static String SECTIONTAG_HEAD = "<!-- @@";
	private final static String SECTIONTAG_TAIL = "@@ -->";

	private final static int DELIMITER_LEN = 2;
	private final static int SECTIONTAG_HEAD_LEN = 7;
	private final static int SECTIONTAG_TAIL_LEN = 6;

	private final static int MAX_FIELDS = 31;

	// Create empty, initialized template object
	public Template() {
		init();
	}
	
	// Create using template byte-array
	public Template(byte[] content) throws Exception{
		setContent(content);
	}
	

	// Create using template file
	public Template(String filename) throws Exception {
		byte[] bytes = null;
		try 	{
			RandomAccessFile raf = new RandomAccessFile(filename, "r");
			int len = (int) raf.length();
			bytes = new byte[len];
			raf.read(bytes);
			raf.close();
		} catch (Exception e) {
			throw (new Exception("Template(filename): "+e));
		}
		setContent(bytes);
	}

	// Copy ctor
	public Template(Template srctpl)
	{
		init ();
		
		node tail = head;
		node currNode = srctpl.head;
		section lastSection = firstSection;
		section currSection = srctpl.firstSection.nextSection;

		for (; currSection != null; currSection = currSection.nextSection)
		{
			// Copy part before section
			while (currNode != currSection.preceding)
			{
				currNode = currNode.next;
				tail.next = new node();
				tail = tail.next;
				if (currNode.val.shared)
					tail.val = _produceField(((field)currNode.val).key);
				else
					tail.val = new cell();
				tail.val.val = currNode.val.val;
			}

			// Create section entry
			lastSection.nextSection = _produceSection(currSection.key);
			lastSection = lastSection.nextSection;
			lastSection.preceding = tail;
			lastSection.tpl = new Template(currSection.tpl);
			lastSection.tpl.parent = this;
			
			// Copy added content
			if (currSection.tpl.addedHead.next != null)
			{
				int len = 0;
				for (node n=currSection.tpl.addedHead.next; n!=null; n=n.next)
					len += n.val.val.length();
				lastSection.tpl.addedTail.next = new node();
				lastSection.tpl.addedTail = lastSection.tpl.addedTail.next;
				lastSection.tpl.addedTail.val = new cell();

				// Make content string
				StringBuffer sb = new StringBuffer(len);
				for (node n=currSection.tpl.addedHead.next; n!=null; n=n.next)
					sb.append(n.val.val);

				lastSection.tpl.addedTail.val.val = sb.toString();
			}
		}
		// Copy rest
		while ((currNode = currNode.next) != null)
		{
			tail.next = new node();
			tail = tail.next;
			if (currNode.val.shared)
				tail.val = _produceField(((field)currNode.val).key);
			else
				tail.val = new cell();
			tail.val.val = currNode.val.val;
		}		
	}

   private void init () {
		head = new node();
		addedHead = new node();
		addedTail = addedHead;
		firstSection = new section();
		tpl = this;
		fields = new Object[MAX_FIELDS];
		sections = new Object[MAX_FIELDS];
   }
   
   public void setContent(byte[] content) throws Exception {
   		init ();
		int len = content.length;
		StringBuffer sb = new StringBuffer(len+SECTIONTAG_HEAD_LEN+SECTIONTAG_TAIL_LEN);
		sb.append(SECTIONTAG_HEAD);
		sb.append(new String(content));
		sb.append(SECTIONTAG_TAIL);
		construct(sb.toString(), SECTIONTAG_HEAD_LEN, len+SECTIONTAG_HEAD_LEN);
   }
	
	
	// This is private method, but it shows how the thing gets created using recursion
	// so I let it stay close to top.
	private void construct(String data, int oLast, int oEnd) throws Exception
	{
		node tail = head;
		section lastSection = firstSection;
		int oCurr, oNext;

		while ((oCurr = data.indexOf(DELIMITER, oLast)) != -1 && oCurr < oEnd)
		{
			// Move past delimiter
			oCurr += DELIMITER_LEN;

			// Find matching delimiter
			if ((oNext = data.indexOf(DELIMITER, oCurr)) == -1 || oNext > oEnd)
				throw(new Exception("Template::construct: bad syntax: missing delimiter"));

			// Section
			if (SECTIONTAG_HEAD.regionMatches(0, data, oCurr-SECTIONTAG_HEAD_LEN, SECTIONTAG_HEAD_LEN) 
				&& SECTIONTAG_TAIL.regionMatches(0, data, oNext, SECTIONTAG_TAIL_LEN))
			{
				// If there is any text since last field or section tag
				if (oCurr - oLast > SECTIONTAG_HEAD_LEN)
				{
					tail.next = new node();
					tail = tail.next;
					tail.val = new cell();
					tail.val.val = data.substring(oLast, oCurr-SECTIONTAG_HEAD_LEN);
				}
				String tag = data.substring(oCurr-SECTIONTAG_HEAD_LEN, oNext+SECTIONTAG_TAIL_LEN);
				//System.out.println(tag);
				section s = _produceSection(data.substring(oCurr,oNext));
				oLast = oNext + SECTIONTAG_TAIL_LEN;
				oNext = data.indexOf(tag, oLast);
				if (oNext == -1 || oNext > oEnd)
					throw(new Exception("Template::construct: Bad syntax: Missing section end tag"));
				if (s.preceding != null)
					throw(new Exception("Template::construct: Bad syntax: Duplicate section"));
				s.preceding = tail;
				s.tpl = new Template();
				s.tpl.construct(data, oLast, oNext);
				s.tpl.parent = this;
				lastSection.nextSection = s;
				lastSection = s;
				oLast = oNext + tag.length();			
			}
			// Field
			else
			{
				if (oCurr - oLast > DELIMITER_LEN)
				{
					tail.next = new node();
					tail = tail.next;
					tail.val = new cell();
					tail.val.val = data.substring(oLast, oCurr-DELIMITER_LEN);
				}
				field f = _produceField(data.substring(oCurr, oNext));
				tail.next = new node();
				tail = tail.next;
				tail.val = f;
				oLast = oNext + DELIMITER_LEN;
			}
		}
		if (oLast < oEnd)
		{
			// Save rest of text
			tail.next = new node();
			tail.next.val = new cell();
			tail.next.val.val = data.substring(oLast, oEnd);
		}
	}


	public void reset()
	{
		for (int i=0; i<MAX_FIELDS; i++)
		{
			for (field f = (field)fields[i]; f != null; f = f.next)
				f.val = "";
		}
		for (section s = firstSection.nextSection; s != null; s = s.nextSection)
		{
			s.tpl.addedHead.next = null;
			s.tpl.addedTail = s.tpl.addedHead;
			s.tpl.reset();
		}
	}

	public void setField(String key, String val)
	{
		field f = tpl._getField(key);
		if (f != null)
			f.val = val;
	}

	public void setField(String key, int val)
	{
		field f = tpl._getField(key);
		if (f != null)
			f.val = Integer.toString(val);
	}

	public void setField(String key, double val)
	{
		field f = tpl._getField(key);
		if (f != null)
			f.val = Double.toString(val);
	}

	public void setFieldGlobal(String key, String val)
	{
		setField(key, val);
		for (section s=firstSection.nextSection; s!=null; s=s.nextSection)
			s.tpl.setFieldGlobal(key, val);
	}

	public void setFieldGlobal(String key, int val)
	{
		setFieldGlobal(key, Integer.toString(val));
	}

	public void setFieldGlobal(String key, double val)
	{
		setFieldGlobal(key, Double.toString(val));
	}

	public void setFieldFromFile(String key, String filename) throws Exception
	{
		field f = _getField(key);
		if (f != null)
		{
			try
			{
				RandomAccessFile raf = new RandomAccessFile(filename, "r");
				byte[] bytes = new byte[(int)raf.length()];
				raf.read(bytes);
				raf.close();
				f.val = new String(bytes);
			}
			catch (Exception e)
			{
				throw(new Exception("Template::setFieldFromFile: "+e));
			}
		}
	}

	public void selectSection(String key)
	{
		section s = tpl._getSection(key);
		if (s != null)
		{
			tpl.tpl = s.tpl;
			tpl = tpl.tpl;
		}
	}

	public void deselectSection()
	{
		if (tpl != this)
			tpl = tpl.parent;
	}

	public void appendSection()
	{
		if (tpl == this)
			return;
		node currNode = tpl.head;
		for (section currSection=tpl.firstSection.nextSection; currSection!=null; currSection=currSection.nextSection)
		{
			while (currNode != currSection.preceding)
			{
				currNode = currNode.next;
				tpl.addedTail.next = new node();
				tpl.addedTail = tpl.addedTail.next;
				tpl.addedTail.val = new cell();
				tpl.addedTail.val.val = currNode.val.val;
			}
			if (currSection.tpl.addedHead.next != null)
			{
				tpl.addedTail.next = currSection.tpl.addedHead.next;
				tpl.addedTail = currSection.tpl.addedTail;
				currSection.tpl.addedHead.next = null;
				currSection.tpl.addedTail = currSection.tpl.addedHead;
			}
		}
		while ((currNode = currNode.next) != null)
		{
			tpl.addedTail.next = new node();
			tpl.addedTail = tpl.addedTail.next;
			tpl.addedTail.val = new cell();
			tpl.addedTail.val.val = currNode.val.val;
		}
	}

	public void attachSection(String key)
	{
		selectSection(key);
		appendSection();
		deselectSection();
	}

	public void setSection(String key, String data)
	{
		section s = tpl._getSection(key);
		if (s != null)
		{
			node n = new node();
			n.val = new cell();
			n.val.val = data;
			n.next = s.preceding.next;
			s.preceding.next = n;
			s.preceding = n;
		}
	}

	public void setSectionFromFile(String key, String filename) throws Exception
	{
		section s = tpl._getSection(key);
		if (s != null)
		{
			try
			{
				node n = new node();
				n.val = new cell();
				RandomAccessFile raf = new RandomAccessFile(filename, "r");
				byte[] bytes = new byte[(int)raf.length()];
				raf.read(bytes);
				raf.close();
				n.val.val = new String(bytes);
				n.next = s.preceding.next;
				s.preceding.next = n;
				s.preceding = n;
			}
			catch (Exception e)
			{
				throw(new Exception("Template::setSectionFromFile: "+e));
			}
		}		
	}

	public String getSection(String key)
	{
		section s = tpl._getSection(key);
		if (s == null)
			return "";
		return s.tpl.getContent();
	}

	public String getContent()
	{
		int len = 0;

		// Traverse this' content to get length
		for (node n=head.next; n!=null; n=n.next)
			len += n.val.val.length();
		
		// Traverse all sections to add length of content added to them
		for (section currSection=firstSection.nextSection; currSection!=null; currSection=currSection.nextSection)
		{
			for (node n=currSection.tpl.addedHead.next; n!=null; n=n.next)
				len += n.val.val.length();
		}

		// Traverse to get content in the right order
		node currNode = head;
		StringBuffer sb = new StringBuffer(len);
		for (section currSection=firstSection.nextSection; currSection!=null; currSection=currSection.nextSection)
		{
			while (currNode != currSection.preceding)
			{
				currNode = currNode.next;
				sb.append(currNode.val.val);
			}
			for (node n=currSection.tpl.addedHead.next; n!=null; n=n.next)
				sb.append(n.val.val);			
		}
		while ((currNode = currNode.next) != null)
				sb.append(currNode.val.val);
		return sb.toString();
	}

	/*
	 * Support methods
	 */

	private field _getField(String key)
	{
		field f = (field)fields[Math.abs(key.hashCode() % MAX_FIELDS)];
		while (f != null)
		{
			if (f.key.equals(key))
				return f;
			f = f.next;
		}
		return null;
	}

	private field _produceField(String key)
	{
		int pos = Math.abs(key.hashCode() % MAX_FIELDS);
		field f = (field)fields[pos];
		while (f != null)
		{
			if (f.key.equals(key))
				return f;
			f = f.next;
		}
		f = new field(key);
		f.next = (field)fields[pos];
		fields[pos] = f;
		return f;
	}

	private section _getSection(String key)
	{
		section s = (section)sections[Math.abs(key.hashCode() % MAX_FIELDS)];
		while (s != null)
		{
			if (s.key.equals(key))
				return s;
			s = s.next;
		}
		return null;
	}

	private section _produceSection(String key)
	{
		int pos = Math.abs(key.hashCode() % MAX_FIELDS);
		section s = (section)sections[pos];
		while (s != null)
		{
			if (s.key.equals(key))
				return s;
			s = s.next;
		}
		s = new section();
		s.key = key;
		s.next = (section)sections[pos];
		sections[pos] = s;
		return s;
	}

	/*
	 * Internal data types
	 */

	private class cell
	{
		public String val;
		public boolean shared;

		public cell()
		{
			shared = false;
		}
	}

	private class node
	{
		public cell val;
		public node next;
	}

	public class field extends cell
	{
		public String key;
		public field next;

		public field(String Key)
		{
			key = Key;
			val = "";
			shared = true;
		}
	}

	private class section
	{
		public String key;
		public Template tpl;
		public node preceding;
		public section next;
		public section nextSection;
	}

	/*
	 * Members
	 */

	private node head;
	private node addedHead;
	private node addedTail;
	private Object[] fields;
	private Object[] sections;
	private section firstSection;
	private Template tpl;
	private Template parent;
}
