/*
Very Quick Wiki - WikiWikiWeb clone
Copyright (C) 2001-2002 Gareth Cronin

This program is free software; you can redistribute it and/or modify
it under the terms of the latest version of the GNU Lesser General
Public License as published by the Free Software Foundation;

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program (gpl.txt); if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package vqwiki.utils;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;
import vqwiki.Environment;
import vqwiki.WikiBase;
import vqwiki.WikiException;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.*;
import java.text.DateFormat;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.net.URL;
import java.net.MalformedURLException;

public class Utilities {

  private static final Logger logger = Logger.getLogger(Utilities.class);

  private static final int STATE_NO_ENTITY = 0;
  private static final int STATE_AMPERSAND = 1;
  private static final int STATE_AMPERSAND_HASH = 2;
  private static final String URLNOTALLOWED = "<>&|`\\\"";
  private static SimpleDateFormat dateFormat = new SimpleDateFormat ("yyyy.MM.dd.HH.mm.ss");


  public static boolean checkURLChars(String urlString) {
  	return (getWrongURLChars(urlString).length()==0);
  }
  
  public static String getWrongURLChars(String urlString) {
  	if (urlString==null) return "";
	StringBuffer sb = new StringBuffer();
  	for (int i = 0; i < urlString.length(); i++) {
  		if (URLNOTALLOWED.indexOf(urlString.substring(i, i+1))>=0) {
  			sb.append(urlString.substring(i, i+1));
  		}
  	}
  	return sb.toString();  	
  }
  
	public static String encodeSafeURL(String topic) {
		if (topic == null) {return null; }
		StringBuffer sb = new StringBuffer ();
		for (int i = 0; i < topic.length(); i++) {
			char c = topic.charAt(i);
			int ci = (int) c;
			if (c == ' ') {sb.append('+');}
			else
			if ((c == '!') ||
                (c == '.') ||
				(c == '-') ||
				(c == '_'))
				{sb.append(c);}
			else
			if ((ci <=31) ||
				((ci >= 33) && (ci <=47)) ||
			    ((ci >= 58) && (ci <=64)) ||
			    ((ci >= 91) && (ci <=96)) ||
			    (ci >= 123)) {
				    String hexString = "0" + Integer.toHexString(ci).toUpperCase();
				    hexString = hexString.substring(hexString.length()-2);
			    	sb.append("($");
			    	sb.append(hexString);
			    	sb.append(")");
			    }
			else {
				sb.append(c);
			}
		}
		
		return sb.toString();
	}
  
  /**
   * Returns true if the given collection of strings contains the given string where the case
   * of either is ignored, if the filesystem ignores the case 
   * @param collection collection of {@link String}s
   * @param string string to find
   * @return true if the string is in the collection with no regard to case
   */
  public static boolean containsFileNameOSDependant(Collection collection, String string){
  	String os = System.getProperty("os.name");
  	boolean ignoreCase;
  	if (os.startsWith("Linux")) {
  		ignoreCase = false;
  	} else if (os.startsWith("Windows")){
  		ignoreCase = true;	
  	} else {
  		ignoreCase = true;
  	}
  	
    for (Iterator iterator = collection.iterator(); iterator.hasNext();) {
      String s = (String) iterator.next();
      if ((ignoreCase & s.equalsIgnoreCase(string))|
      	  (!ignoreCase & s.equals(string)))
      	return true;
    }
    return false;
  }

  public static Cookie createUsernameCookie(String username) {
    Cookie c = new Cookie("username", username);
    c.setMaxAge(Environment.getInstance().getIntSetting(Environment.PROPERTY_COOKIE_EXPIRE));
    return c;
  }


  /**
   * Converts HTML integer unicode entities into characters
   * @param text
   * @return
   * @throws java.lang.Exception
   */
  public static String convertEntities(String text) throws Exception {
    StringReader in = new StringReader(text);
    StringBuffer buffer = new StringBuffer();
    StringBuffer current = new StringBuffer();
    int nextByte;
    int entityMode = STATE_NO_ENTITY;
    while (-1 != (nextByte = in.read())) {
      char nextChar = (char) nextByte;
      if (entityMode == STATE_NO_ENTITY) {
        if (nextChar != '&')
          buffer.append(nextChar);
        else {
          entityMode = STATE_AMPERSAND;
          current.append(nextChar);
        }
      }
      else if (entityMode == STATE_AMPERSAND) {
        if (nextChar != '#') {
          buffer.append(current);
          buffer.append(nextChar);
          current = new StringBuffer();
          entityMode = STATE_NO_ENTITY;
        }
        else {
          entityMode = STATE_AMPERSAND_HASH;
          current.append(nextChar);
        }
      }
      else {
        if (nextChar == ';') {
          String stringValue = current.substring(2, current.length());
          logger.debug("entity found: " + stringValue);
          buffer.append((char) Integer.parseInt(stringValue));
          current = new StringBuffer();
          entityMode = STATE_NO_ENTITY;
        }
        else if (Character.isDigit(nextChar)) {
          current.append(nextChar);
        }
        else {
          buffer.append(current);
          current = new StringBuffer();
          entityMode = STATE_NO_ENTITY;
        }
      }
    }
    return buffer.toString();
  }

  /**
   * Localised
   */
  public static String formatDate(Date date) {
    return DateFormat.getDateInstance().format(date);
  }

  public static String formatDateTime(Date date) {
    return DateFormat.getDateTimeInstance().format(date);
  }


  /**
   * Convert the filename-friendly date format back to a Java date
   */
  public static Date convertFileFriendlyDate(String filename) {
    return dateFormat.parse(filename, new ParsePosition(filename.length()-19));
  }

  /**
   * Convert Java date to a file-friendly date
   */
  public static String fileFriendlyDate(Date date) {
  	return dateFormat.format(date);
  }

  /**
   * Convert 3 consecutive spaces to a tab character
   */
  public static String convertTabs(String text) {
    boolean preformatted = false;
    String converted = "";
    int spaces = 0;
    int ats = 0;
    int linefeeds = 0;
    for (int i = 0; i < text.length(); i++) {
      if (text.charAt(i) == ' ')
        spaces++;
      else if (text.charAt(i) == '@')
        ats++;
      else if (text.charAt(i) == '\n')
        linefeeds++;
      else if (text.charAt(i) != '\r') {
        linefeeds = 0;
        spaces = 0;
        ats = 0;
      }
      else {
        spaces = 0;
        ats = 0;
      }
      converted += text.charAt(i);
      if (ats == 4) {
        preformatted = true;
        ats = 0;
        linefeeds = 0;
      }
      if (linefeeds == 2) {
        linefeeds = 0;
        preformatted = false;
      }
      if (spaces == 3 && (!preformatted)) {
        converted = converted.substring(0, converted.length() - 3) + "\t";
        spaces = 0;
      }
    }
    return converted;
  }

  public static String getDirectoryFromPath(String path) {
    logger.debug("getDirectoryFromPath: " + path);
    return path.substring(0, 1 + path.lastIndexOf('/'));
  }

  public static String sep() {
    return System.getProperty("file.separator");
  }

 
  /*
   * Extracts the virtual wiki from the URL path.  Defaults to the default
   * wiki if there's trouble.
   *
   */
  public static String extractVirtualWiki(HttpServletRequest request) throws WikiException {
//  String path = HttpUtils.getRequestURL(request).toString();
    String path = request.getRequestURL().toString();
    return extractVirtualWiki(path);
  }

  /**
   * returns the full virtual wiki hostname port and context (including the virtual wiki context)
   * @param request
   * @return
   */
  public static String extractVirtualWikiContext(HttpServletRequest request) {
    String path = request.getRequestURL().toString();
    // path = path.substring(0, path.lastIndexOf('/'));
    return path;
  }

  /**
   * returns the full virtual wiki hostname port and context (including the virtual wiki context) but
   * replaces the hostname with the given one
   * @param request
   * @return
   */
  public static String extractVirtualWikiContext(HttpServletRequest request, String useHostname) throws MalformedURLException {
	if (useHostname == null) return extractVirtualWikiContext (request);
    String path = request.getRequestURL().toString();
    
    // String originalRequestContext = path.substring(0, path.lastIndexOf('/'));
    String originalRequestContext = path;
    URL originalUrl = new URL(originalRequestContext);
    URL replacedUrl = new URL(
        originalUrl.getProtocol(),
        useHostname,
        originalUrl.getPort(),
        originalUrl.getFile()
    );
    return replacedUrl.toString();
  }

  public static String extractVirtualWiki(String path) throws WikiException {

    logger.debug("path: " + path);
    StringTokenizer tokenizer = new StringTokenizer(path, "/");

    // if there's no tokens, use default
    if (tokenizer.countTokens() == 0)
      return WikiBase.DEFAULT_VWIKI;

    String[] tokens = new String[tokenizer.countTokens()];
    int i = 0;
    int jspPosition = -1;
    while (tokenizer.hasMoreTokens()) {
      tokens[i] = tokenizer.nextToken();
      logger.debug("tokens[" + i + "]: " + tokens[i]);
      if (tokens[i].equalsIgnoreCase("jsp"))
        jspPosition = i;
      i++;
    }
    logger.debug("jspPosition: " + jspPosition);

    // We didn't find "jsp" in the path
    if (jspPosition == -1) {
      // servlet name only (application is root level context)
      if (i == 1)
        return WikiBase.DEFAULT_VWIKI;

      else if (isAVirtualWiki(tokens[i - 2]))
        return tokens[i - 2];
      else {
        String errorMessage = "The virtual wiki that you have chosen " +
            "does not exist. Verify the web address that you entered. " +
            "If the problem continues, contact your wiki administrator";
        throw new WikiException(errorMessage);
      }
    }

    // handle the case where we have a "jsp" token in the URL

    // jsp is first in path (last element is always the servlet/jsp)
    // if we have only the servlet/jsp name assume default virtual wiki
    else if (jspPosition == 0 || i == 1)
      return WikiBase.DEFAULT_VWIKI;

    else if (isAVirtualWiki(tokens[jspPosition - 1]))
      return tokens[jspPosition - 1];

    // if we come up with anything not in the vwiki list, we use the default
    else
      return WikiBase.DEFAULT_VWIKI;
  }

  /*
   * checks the string parameter against there virtual wiki list
   */

  public static boolean isAVirtualWiki(String virtualWiki) {
    try {
      WikiBase wikibase = WikiBase.getInstance();
      Iterator wikilist = wikibase.getVirtualWikiList().iterator();
      while (wikilist.hasNext()) {
        if (virtualWiki.equals((String) wikilist.next()))
          return true;
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
    return false;
  }

  public static String getUserFromRequest(HttpServletRequest request) {
    Cookie[] cookies = request.getCookies();
    if (cookies == null) return null;
    if (cookies.length > 0)
      for (int i = 0; i < cookies.length; i++)
        if (cookies[i].getName().equals("username"))
          return cookies[i].getValue();
    return null;
  }

  public static boolean checkValidTopicName(String name) {
    if (name.indexOf(':') >= 0 || name.indexOf('/') >= 0 || name.indexOf('.') >= 0 ||
        name.indexOf("%3a") >= 0 || name.indexOf("%2f") >= 0 || name.indexOf("%2e") >= 0)
      return false;
    return true;
  }

  /**
   * Return a String from the ApplicationResources. The ApplicationResources contains all text,
   * links, etc. which may be language specific and is located in the classes folder.
   * @param key The key to get.
   * @param locale The locale to use.
   * @return String containing the requested text.
   */
  public static String resource(String key, Locale locale, String defaultMessage) {
  	String message = null;
  	try {
  		ResourceBundle messages = getMessages(locale);
  		message = messages.getString(key);
  	} catch (MissingResourceException e) {
  		if (defaultMessage != null) {
  			message = defaultMessage;
  		} else {
  			throw e;
  		}
  	}
  	return message;
  }
  
  /**
   * Return a String from the ApplicationResources. The ApplicationResources contains all text,
   * links, etc. which may be language specific and is located in the classes folder.
   * @param key The key to get.
   * @param locale The locale to use.
   * @return String containing the requested text.
   */
  public static String resource(String key, Locale locale) {
  	return resource (key, locale, null);
  }

  public static ResourceBundle getBlackList (String virtualWiki) {
  	ResourceBundle blackList= null;
  	String blackListFile = null;
  	try {
  		blackListFile = resource("virtualwiki." + virtualWiki + ".blacklist", Locale.getDefault());
  	} catch (MissingResourceException e) {}
  	if (blackListFile == null) return blackList;
  	try {
  		blackList = ResourceBundle.getBundle(blackListFile);
  	} catch (MissingResourceException e) {
  		logger.debug("BlackList not found: " + blackListFile);
  	}
  	return blackList;
  }

  public static ResourceBundle getWhiteList (String virtualWiki) {
  	ResourceBundle whiteList = null;
  	String whiteListFile = null;
  	try {
  		whiteListFile = resource("virtualwiki." + virtualWiki + ".whitelist", Locale.getDefault());
  	} catch (MissingResourceException e) {}
  	if (whiteListFile == null) return whiteList;
  	try {
  		whiteList = ResourceBundle.getBundle(whiteListFile);
  	} catch (MissingResourceException e) {
  		logger.debug("WhiteList not found: " + whiteListFile);
  	}
  	return whiteList;
  }
  
  /**
   * Get messages for the given locale
   * @param locale locale
   * @return
   */
 
  public static ResourceBundle getMessages() {
  	return new VQResourceBundle();
  }
  /**
   * Get messages for the given locale
   * @param locale locale
   * @return
   */
 
  public static ResourceBundle getMessages(Locale locale) {
  	return new VQResourceBundle(locale);
  }
 
  public static String convertLineFeeds(String text) {
    StringBuffer buffer = new StringBuffer();
    boolean rSeen = false;
    for (int i = 0; i < text.length(); i++) {
      char c = text.charAt(i);
      if (c == '\n') {
        buffer.append("\r\n");
      }
      else if (c == '\r') {
        //do nothing
      }
      else {
        buffer.append(c);
      }
    }
    return buffer.toString();
  }

  /**
   * Read a file from the file system
   * @param file The file to read
   * @return a Stringbuffer with the content of this file or an empty StringBuffer, if an error has occured
   */
  public static StringBuffer readFile(File file) {
    char[] buf = new char[1024];
    StringBuffer content = new StringBuffer((int)file.length());
    try {
      Reader in = new BufferedReader(new InputStreamReader(new FileInputStream(file),Environment.getInstance().getFileEncoding()));
      int numread = 0;
      while((numread=in.read(buf))!=-1) {
	  content.append(buf,0,numread);
      }
      in.close();
    }
    catch (IOException e1) {
      e1.printStackTrace();
    }
    return content;
  }

  public static final void copyInputStream(InputStream in, OutputStream out)
      throws IOException {
    byte[] buffer = new byte[1024];
    int len;

    while ((len = in.read(buffer)) >= 0)
      out.write(buffer, 0, len);
    in.close();
    out.close();
  }

  public static void unzip(File zipFileToOpen, File outputDirectory) {
    Enumeration entries;
    ZipFile zipFile;
    try {
      zipFile = new ZipFile(zipFileToOpen);
      entries = zipFile.entries();
      while (entries.hasMoreElements()) {
        ZipEntry entry = (ZipEntry) entries.nextElement();
        if (entry.isDirectory()) {
          logger.debug("Extracting directory: " + entry.getName());
          // This is not robust, just for demonstration purposes.
          File file = new File(outputDirectory, entry.getName());
          file.mkdir();
        }
      }
      entries = zipFile.entries();
      while (entries.hasMoreElements()) {
        ZipEntry entry = (ZipEntry) entries.nextElement();
        if (!entry.isDirectory()) {
          logger.debug("Extracting file: " + entry.getName());
          File outputFile = new File(outputDirectory, entry.getName());
          copyInputStream(zipFile.getInputStream(entry),
                          new BufferedOutputStream(new FileOutputStream(outputFile)));
        }
      }
      zipFile.close();
    }
    catch (IOException ioe) {
      logger.error("Unzipping error: " + ioe);
      ioe.printStackTrace();
      return;
    }
  }

  /**
   * Use standard factory to create DocumentBuilder and parse a file
   */
  public static Document parseDocumentFromFile(String fileName) throws Exception {
    File file = new File(fileName);
    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    DocumentBuilder builder = factory.newDocumentBuilder();
    return builder.parse(file);
  }

  public static boolean isAdmin(HttpServletRequest request) {
    HttpSession session = request.getSession();
    String value = (String) session.getAttribute("admin");
    return "true".equals(value);
  }

  /**
   * Replaces occurences of the find string with the replace string in the given text
   * @param text
   * @param find
   * @param replace
   * @return the altered string
   */
  
  public static String replaceString(String text, String find, String replace) {
    int findLength = find.length();
    StringBuffer buffer = new StringBuffer();
    int i;
    for (i = 0; i < text.length() - find.length() + 1; i++) {
      String substring = text.substring(i, i + findLength);
      if (substring.equals(find)) {
        buffer.append(replace);
        i += find.length() - 1;
      }
      else {
        buffer.append(text.charAt(i));
      }
    }
    buffer.append(text.substring(text.length() - (text.length() - i)));
    return buffer.toString();
  }
  
  /*
  private String replaceString(String source, String searchStr, String repStr) {
  	if ((source != null) && (searchStr != null) && (repStr != null)) {
      	while (source.indexOf(searchStr)>=0) {
      		int pos = source.indexOf(searchStr);
      		source = source.substring(0, pos) + repStr + source.substring(pos+searchStr.length());
      	}
  	}
  	return source;
  }
  */
  /**
   * Returns any trailing . , ; : characters on the given string
   * @param text
   * @return empty string if none are found
   */
  public static String extractTrailingPunctuation(String text) {
    StringBuffer buffer = new StringBuffer();
    for (int i = text.length() - 1; i >= 0; i--) {
      char c = text.charAt(i);
      if (c == '.' || c == ';' || c == ',' || c == ':') {
        buffer.append(c);
      }
      else {
        break;
      }
    }
    if (buffer.length() == 0)
      return "";
    buffer = buffer.reverse();
    return buffer.toString();
  }

/**
 * Converts arbitrary string into string usable as file name.
 */
  public static String encodeSafeFileName(String name) {
  	/* old version
     StringTokenizer st = new StringTokenizer(name,"%"+File.separator,true);
     StringBuffer sb = new StringBuffer(name.length());
     while (st.hasMoreTokens()) {
         String token = st.nextToken();
         if(File.separator.equals(token)||"%".equals(token)) {
            sb.append(token);
         } else {
            sb.append(JSPUtils.encodeURL(token,"utf-8"));
         }
     }
     return sb.toString();
     */
  	return encodeSafeURL(name);
  }
  
  public static String decodeSafeURL (String name) {
  	StringBuffer sb = new StringBuffer();
  	int i = 0; 
  	while (i < name.length()) {
  		char c = name.charAt(i);
  		if (c == '+') {
  			sb.append(' ');
  			i++;  			
  		}
		else if (c!='(') {
  			sb.append(c);
  			i++;
  		} else {
  			String hex = name.substring(i + 2, i + 4);
  			sb.append((char)Integer.parseInt(hex, 16));
  			i = i + 5;
  		}
  	}
  	return sb.toString();
  }
/**
 * Converts back file name encoded by encodeSafeFileName().
 */
  public static String decodeSafeFileName(String name) {
  	/* old version
    return JSPUtils.decodeURL(name,"utf-8");
  */
  	return decodeSafeURL(name);
  }
  
  public static String encodeHTMLChars (String s) {
  	if (s == null) return null;
  	StringBuffer sb = new StringBuffer ();
  	for (int i = 0; i < s.length(); i++) {
  		char c = s.charAt(i);
  		int ci = (int) c;
  		if ((ci < 32) | (ci>127)) {
  			sb.append("&#");
  			sb.append(ci);
  			sb.append(";");
  		} else {
  			sb.append (c);
  		}
  	}
  	return sb.toString();
  }
  /**
   * Converts CamelCase to seperate words.
   * @author cmeans
   *
   */
  public static String separateWords (String text) 
  {
    // Do not try to separateWords if there are spaces in the text.
    if (text.indexOf(" ") != -1)
      return text ;
      
    // Allocate enough space for the new string, plus
    // magic number (5) for a guess at the likely max
    // number of words.
    StringBuffer sb = new StringBuffer (text.length () + 5) ;
   
    int offset = 0 ;  // points to the start of each word.
    
    // Loop through the CamelCase text, at each capital letter
    // concatenate the previous word text to the result
    // and append a space.
    // The first character is assumed to be a "capital".
    for (int i = 1 ;
         i < text.length () ;
         i++)
      if (   (text.charAt (i) >= 'A')
          && (text.charAt (i) <= 'Z')) 
      {
        // Append the current word and a trailing space.
        sb.append (text.substring (offset, 
                                   i))
          .append (" ") ;
              
        // Start of the next word.
        offset = i ;
      }

    // Append the last word.
    sb.append (text.substring (offset)) ;

    return sb.toString () ;
  }

  /**
   * Parse DOM document from XML in input stream
   * @param xmlIn
   * @return
   * @throws IOException
   * @throws SAXException
   * @throws ParserConfigurationException
   */
  public static Document parseDocumentFromInputStream(InputStream xmlIn) throws IOException, SAXException,
      ParserConfigurationException {
    DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
    DocumentBuilder builder = factory.newDocumentBuilder();
    return builder.parse(xmlIn);
  }

  public static void printStackTrace () {
    try {
    	throw new Exception ();
    } catch (Exception e) {
    	e.printStackTrace();
    }
  }

  public static String extractCharacterEncoding (HttpServletResponse response) {
  	return extractCharacterEncoding(null, response, false);
  }

  public static String extractCharacterEncoding (HttpServletRequest request) {
  	return extractCharacterEncoding(request, null, true);
  }
  
  public static String extractCharacterEncoding (HttpServletRequest request, HttpServletResponse response) {
  	return extractCharacterEncoding(request, response, true);
  }

  public static String extractCharacterEncoding (HttpServletRequest request, HttpServletResponse response, boolean requestFirst) {
  	if (requestFirst) {
  		if ((request != null) && request.getCharacterEncoding()!= null) return request.getCharacterEncoding();
  		if ((response != null) && response.getCharacterEncoding()!= null) return response.getCharacterEncoding();
  	} else {
  		if ((response != null) && response.getCharacterEncoding()!= null) return response.getCharacterEncoding();
  		if ((request != null) && request.getCharacterEncoding()!= null) return request.getCharacterEncoding();
  	}
  	return JSPUtils.DEFAULTENCODING;
  }
}
