package vqwiki.lex;

/*
Very Quick Wiki - WikiWikiWeb clone
Copyright (C) 2001-2003 Gareth Cronin

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program (gpl.txt); if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

import java.io.*;
import org.apache.log4j.*;
import vqwiki.lex.Lexer;
import vqwiki.Environment;
import vqwiki.WikiBase;

%%

%public
%type String
%unicode
%extends vqwiki.lex.Lexer
%class LayoutLex

%init{
	yybegin( NORMAL );
%init}

%eofval{
    cat.debug( "Ending at list level: " + listLevel );
    if( yystate() == LIST ){
      yybegin( NORMAL );
      StringBuffer buffer = new StringBuffer();
      for( int i = listLevel; i > 0; i-- ){
        buffer.append( "</li>\n" );
        if( ordered )
          buffer.append( "</ol>\n" );
        else
          buffer.append( "</ul>\n" );
      }
    return buffer.toString();
    }
    if( yystate() == TABLE ){
      yybegin( NORMAL );
      return "</tr>\n</table>\n";
    }

	return null;
%eofval}

%{
  protected boolean allowHtml;
  protected int listLevel;
  protected boolean ordered = false;
  protected boolean definition = false;
  protected static Logger cat = Logger.getLogger( LayoutLex.class );
  protected String virtualWiki;

	protected boolean exists( String topic ){
	  try{
	    return WikiBase.getInstance().exists( virtualWiki, topic );
	  }catch( Exception err ){
	    cat.error( err );
	  }
	  return false;
	}

  public void setVirtualWiki( String vWiki ){
    this.virtualWiki = vWiki;
  }

%}

newline	= (\n|\r\n)
whitespace = ([\t\ \r\n])
hr = (----)
noformat	=	(__)
tableboundary = (####{newline})
tablecell =	([^#\n]+##)
externalstart = (\[<[^>]+>\])
externalend = (\[<\/[:letter:]+>\])

%state NORMAL, OFF, PRE, LIST, TABLE, EXTERNAL, GLOSSAR

%%

<NORMAL, PRE>\\{noformat}	{
  cat.debug( "escaped double backslash" );
  return "\\__";
}

<NORMAL, PRE>{noformat}	{
  cat.debug( "Format off" );
  yybegin( OFF );
  return "__";
}

<OFF>{noformat} {
  cat.debug( "Format on" );
  yybegin( NORMAL );
  return "__";
}

<NORMAL, PRE>{externalstart} {
  cat.debug( "external" );
  yybegin( EXTERNAL );
  return yytext();
}

<EXTERNAL>{externalend} {
  cat.debug( "external end");
  yybegin( NORMAL );
  return yytext();
}

<NORMAL>({hr}{newline}) {
  cat.debug( "{hr}" );
  return "\n<hr class=\"inner\"/>\n";
}

<NORMAL>(\t+[$=].+{newline})|({newline}\t+[$=].+{newline}) {
  yybegin (GLOSSAR);
  String text = yytext().trim();
  cat.debug( "first glossar item: " + text );
  definition = (text.charAt(0)=='=');
  StringBuffer buffer = new StringBuffer();
  buffer.append( "<dl class=\"inner\">\n" );
  if(definition)
    buffer.append( "<dd class=\"inner\">\n" );
  else
    buffer.append( "<dt class=\"inner\">\n" );
  buffer.append( text.substring(1).trim() );
  if(definition)
    buffer.append( "</dd>\n" );
  else
    buffer.append( "</dt>\n" );
  return buffer.toString();
}

<GLOSSAR>\t+[$=].+{newline} {
  String text = yytext().trim();
  cat.debug( "first glossar item: " + text );
  definition = (text.charAt(0)=='=');
  StringBuffer buffer = new StringBuffer();
  if(definition)
    buffer.append( "<dd class=\"inner\">\n" );
  else
    buffer.append( "<dt class=\"inner\">\n" );
  buffer.append( text.substring(1).trim() );
  if(definition)
    buffer.append( "</dd>\n" );
  else
    buffer.append( "</dt>\n" );
  return buffer.toString();
}

<NORMAL>(\t+[*#].+{newline})|({newline}\t+[*#].+{newline}) {
  yybegin( LIST );
  String text = yytext().trim();
  cat.debug( "first list item: " + text );
  int count = 0;
  ordered = false;

  cat.debug( "First char='" + text.charAt(0) + "'" );
  if( text.charAt( 0 ) == '#' )
    ordered = true;

  listLevel = 1;
  StringBuffer buffer = new StringBuffer();
  if( ordered )
    buffer.append( "<ol class=\"inner\">\n" );
  else
    buffer.append( "<ul class=\"inner\">\n" );
  buffer.append( "<li class=\"inner\">\n" );
  buffer.append( text.substring( count + 1 ).trim() );
  return buffer.toString();
}

<LIST>\t+[*#].+{newline} {
  String text = yytext();
  cat.debug( "another list item: " + text );
  int count = 0;
  for( int i = 0; i < text.length(); i++ ){
    if( text.charAt( i ) == '\t' )
      count++;
    else
      break;
  }
  StringBuffer buffer = new StringBuffer();
  if( count > listLevel ){
    if( ordered )
      buffer.append( "\n<ol class=\"inner\">\n" );
    else
      buffer.append( "\n<ul class=\"inner\">\n" );
    listLevel = count;      
  }
  else if( count < listLevel ){
    for (int i = listLevel; i > count; i--) {
      buffer.append( "</li>\n" );
      if( ordered )
        buffer.append( "</ol>\n" );
      else
        buffer.append( "</ul>\n" );
    }
    if ((listLevel-count)>1) {
      buffer.append( "</li>\n" );
    }
    listLevel = count;
  }
  else{
    buffer.append( "</li>\n" );
  }
  buffer.append( "<li class=\"inner\">\n" );
  buffer.append(text.substring( count + 1 ).trim());
  return buffer.toString();
}

<NORMAL>{tableboundary} {
  cat.debug( "table start" );
  yybegin( TABLE );
  return "<table class=\"inner\">\n<tr class=\"inner\">\n";
}

<TABLE>{tablecell}{newline}{tableboundary} {
  cat.debug("table end");
  yybegin(NORMAL);
  String text = yytext().trim();
  StringBuffer buffer = new StringBuffer();
  buffer.append ("<td class=\"inner\">\n");
  buffer.append (text.substring(0, text.indexOf("##")));
  buffer.append ("</td>\n");
  buffer.append ("</tr>\n</table>\n");
  return buffer.toString();
}

<TABLE>{newline} {
  cat.debug( "tablerowend" );
  return "</tr>\n<tr class=\"inner\">\n";
}

<TABLE>{tablecell} {
  cat.debug( "tablecell" );
  String text = yytext();
  StringBuffer buffer = new StringBuffer();
  buffer.append( "<td class=\"inner\">\n" );
  buffer.append( text.substring( 0, text.length() - 2 ) );
  buffer.append( "</td>\n" );
  return buffer.toString();
}

<LIST>{newline} {
  cat.debug( "end of list" );
  yybegin( NORMAL );
  StringBuffer buffer = new StringBuffer();
  for( int i = listLevel; i > 0; i-- ){
    buffer.append( "</li>\n" );
    if( ordered )
      buffer.append( "</ol>\n" );
    else
      buffer.append( "</ul>\n" );
  }
  return buffer.toString();
}

<GLOSSAR>{newline} {
  cat.debug( "end of glossar" );
  yybegin( NORMAL );
  StringBuffer buffer = new StringBuffer();
  buffer.append( "</dl>\n" );
  return buffer.toString();
}

<PRE>{newline}{newline} {
  cat.debug( "{newline}x2 leaving pre" );
	yybegin( NORMAL );
  return "</pre>\n";
}

<NORMAL, OFF>{newline} {
  cat.debug( "{newline}" );
  StringBuffer buffer = new StringBuffer();
  for( int i = 0; i < Environment.getInstance().getIntSetting( Environment.PROPERTY_NEW_LINE_BREAKS ); i++ )
    buffer.append( "<br/>" );
  buffer.append( "\n" );
  return buffer.toString();
}

<NORMAL>(@@@@{newline}) {
  cat.debug( "@@@@{newline} entering PRE" );
  yybegin( PRE );
  return "<pre>\n";
}

<NORMAL, OFF, LIST, TABLE>{whitespace} {
  String text = yytext();
  StringBuffer buffer = new StringBuffer();
  for( int i = 0; i < text.length(); i++ ){
    buffer.append( (int)text.charAt(i) );
  }
  cat.debug( "{whitespace} " + buffer.toString() );
  return " ";
}

<PRE, EXTERNAL>{whitespace} {
  cat.debug( "PRE, EXTERNAL {whitespace}" );
  return yytext();
}

<NORMAL, PRE, OFF, LIST, TABLE, EXTERNAL>. {
  cat.debug( ". (" + yytext() + ")" );
  return yytext();
}
