package vqwiki.lex;


/*
Very Quick Wiki - WikiWikiWeb clone
Copyright (C) 2001-2003 Gareth Cronin

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program (gpl.txt); if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

import java.io.*;
import java.util.*;
import org.apache.log4j.*;
import vqwiki.WikiBase;
import vqwiki.utils.*;


%%

%public
%type String
%unicode
%extends vqwiki.lex.Lexer
%class FormatLex

%init{
	yybegin( NORMAL );
%init}

%eofval{
	if( strong ){
	  strong = false;
	  return( "</strong>" );
	}
	if( em ){
	  em = false;
	  return( "</em>" );
	}
	return null;
%eofval}

%{
  protected boolean em, strong, underline, center, table, row, cell, allowHtml, code, h1, h2, h3, color;
  protected int listLevel;
  protected boolean ordered;
  protected boolean isCode = false;
  protected String intersection = "";
  protected int intersectCount = 0;
  protected String intersectMode = null;
  
  private static final String ISECSTART = " {${$${$$${--{";
  private static final String ISECEND = "}--}$$$}$$}$}";
  

%}

newline	= (\n|\r\n)
whitespace = ([\t\ \r\n])
nosquare = ([^\[\]])
tab = (\t)
noformat	=	(__)
ignorenewline = (\\\\{newline})
ignorenewline2 = (\\\\\%%%%(.)+{newline})
intersectstart = (<\$\$-)
intersectstart2 = (<\$\$\[{nosquare}+\]-)
intersectstart3 = (<\$\$\[\]-)
intersectstop = (-\$\$>)
comment = (%%%%{newline})
comment2 = (%%%%(.)+{newline})
externalstart = (\[<[^>]+>\])
externalend = (\[<\/[:letter:]+>\])
entity = (&#[:digit:]+;)
colorstart = (%([a-zA-Z]+|\#[0-9a-fA-F]{6})%)
colorend = (%%)

%state NORMAL, OFF, PRE, EXTERNAL,COMMENT, INTERSECT

%%


<NORMAL, PRE>{intersectstart3} {
 // start <$$[]-
  
  logger.debug( "intersect3 start" );

  if (intersectCount==0) {
  	  intersection = "";
	  intersectMode = "";
	  yybegin (INTERSECT);
  }
  intersectCount++;
  return "";
}

<NORMAL, PRE>{intersectstart2} {
 // start <$$[displayMode]-
  
  logger.debug( "intersect2 start" );

  if (intersectCount==0) {
  	  intersection = "";
	  intersectMode = yytext();
	  int start = intersectMode.indexOf("[");
	  int end = intersectMode.indexOf("]");
	  if ((start >= 0) & (end >= 0)) {
	  	intersectMode = intersectMode.substring(start + 1, end);
	  }
	  yybegin (INTERSECT);
  }
  intersectCount++;
  return "";
}

<NORMAL, PRE>{intersectstart} {
  // start <$$-
  logger.debug( "intersect start" );

  if (intersectCount==0) {
  	  intersection = "";
      intersectMode = "all";
      yybegin (INTERSECT);
  }
  intersectCount++;

  return "";
}

<INTERSECT>{intersectstart3} {
  // another <$$[]-
  intersectCount++;
  intersection+= yytext();
  return "";
}

<INTERSECT>{intersectstart2} {
  // another <$$[displayMode]-
  intersectCount++;
  intersection+= yytext();
  return "";
}

<INTERSECT>{intersectstart} {
  // another <$$-
  intersectCount++;
  intersection+= yytext();
  return "";
}

<INTERSECT>{intersectstop} {
 // $$>
  
  intersectCount--;
  if (intersectCount == 0) {  
  
     Boolean globalDisplay = (Boolean) getParameter("globalDisplay");
     if (globalDisplay == null) globalDisplay = new Boolean (true);
     Boolean globalPrint = (Boolean) getParameter("globalPrint");
     if (globalPrint == null) globalPrint = new Boolean (true);
     Boolean globalExport = (Boolean) getParameter("globalExport");;
     if (globalExport == null) globalExport = new Boolean (true);
     
     boolean localDisplay = false;
     boolean localPrint = false;
     boolean localExport = false;

     StringTokenizer st = new StringTokenizer(intersectMode, ", ");

     while (st.hasMoreTokens()) {
       String token = st.nextToken();
       if (token.trim().toLowerCase().equals("display")) {
         localDisplay = true;
       }
       if (token.trim().toLowerCase().equals("print")) {
         localPrint = true;
       }
       if (token.trim().toLowerCase().equals("export")) {
         localExport = true;
       }
       if (token.trim().toLowerCase().equals("!display")) {
         localPrint = true;
         localExport = true;
       }
       if (token.trim().toLowerCase().equals("!print")) {
         localDisplay = true;
         localExport = true;
       }
       if (token.trim().toLowerCase().equals("!export")) {
         localDisplay = true;
         localPrint = true;
       }
       if (token.trim().toLowerCase().equals("all")) {
         localDisplay = true;
         localPrint = true;
         localExport = true;
       }
    }
         
    globalDisplay = new Boolean(globalDisplay.booleanValue() & localDisplay);
    globalPrint = new Boolean (globalPrint.booleanValue() & localPrint);
    globalExport = new Boolean (globalExport.booleanValue() & localExport);

    boolean calcContent = false;
    switch (displayMode) {
      case WikiBase.DISPLAY: calcContent = globalDisplay.booleanValue(); break;
      case WikiBase.PRINT: calcContent = globalPrint.booleanValue(); break;
      case WikiBase.EXPORT: calcContent = globalExport.booleanValue(); break;
    }  
    if (calcContent) {
      BufferedReader in = new BufferedReader(new StringReader(intersection));
      String content = "";
      try {
        Map parameters = new HashMap();
        parameters.put("decodeSubsection", "false");
        parameters.put("globalDisplay", globalDisplay);
        parameters.put("globalPrint", globalPrint);
        parameters.put("globalExport", globalExport);
        List linkList = new ArrayList ();
        List docList = new ArrayList ();
        content = WikiBase.getInstance().cook(in, virtualWiki, topic, topicsList, linkList, docList, displayModeFullInfo, parameters);
        addLinks (linkList);
        addDocs (docList);
        if (content != "") {
          content = Utilities.encodeSafeFileName(content);
          content = ISECSTART + content + ISECEND;
        }
      	intersection = "";
      	yybegin(NORMAL);
      	return content;
      } catch (Exception e) {}
    }
  	intersection = "";
  	yybegin(NORMAL);
  	return "";
  } else {
	  intersection+= "-$$>";
  }
  return "";
}

<INTERSECT>.|{whitespace} {
  // inside <$$-....-$$>
  logger.debug( ". (" + yytext() + ")" );
  intersection = intersection + yytext();
  return "";
}

<NORMAL, PRE>{comment} {
  // start Comment: %%%%{newline}

  logger.debug( "comment start" );

  yybegin (COMMENT);

  return "";

}

<COMMENT>{comment} {
  // end Comment: %%%%{newline}
  yybegin(NORMAL);
  return "";
}


<COMMENT>.|{whitespace} {
  logger.debug( ". (" + yytext() + ")" );
  return "";
}

<NORMAL, PRE, COMMENT>{ignorenewline} {
  // ignore the following new-line
  return "";
}

<NORMAL, PRE, COMMENT>{ignorenewline2} {
  // ignore the following comment and new-line 
  return "";
}

<NORMAL, PRE>\\{noformat}	{
  // escaped double backslash \\_
  logger.debug( "escaped double backslash" );
  return "\\__";
}

<NORMAL, PRE>{noformat}	{
  // formatted text begin __
  logger.debug( "Format off" );
  yybegin( OFF );
  return "__";
}

<OFF>{noformat} {
  // formatted text end__
  logger.debug( "Format on" );
  yybegin( NORMAL );
  return "__";
}

<NORMAL, PRE>{comment2} {
  // start Comment: %%%%....{newline}

  logger.debug( "comment line" );

  return "\n";

}


<NORMAL, PRE>{externalstart} {
  // external start
  logger.debug( "external" );
  yybegin( EXTERNAL );
  return yytext();
}

<EXTERNAL>{externalend} {
  // external end
  logger.debug( "external end");
  yybegin( NORMAL );
  return yytext();
}

<NORMAL>^\!\!\![^\n]+\!\!\!{newline} {
  // heading 1 !!!
  logger.debug("!!!...!!!");
  return "<h1>" + yytext().substring(3, yytext().substring(3).indexOf('!')+3) + "</h1>";
}

<NORMAL>^\!\![^\n]+\!\!{newline} {
  // heading 2 !!
  logger.debug("!!...!!");
  return "<h2>" + yytext().substring(2, yytext().substring(2).indexOf('!')+2) + "</h2>";
}

<NORMAL>^\![^\n]+\!{newline} {
  // heading 3 !
  logger.debug("!...!");
  return "<h3>" + yytext().substring(1,yytext().substring(1).indexOf('!')+1) + "</h3>";
}

<NORMAL>"'''" {
  // strong '''
  logger.debug( "'''" );
  if( strong ){
    strong = false;
    return( "</strong>" );
  }
  else{
    strong = true;
    return( "<strong>" );
  }
}


<NORMAL>"''"	{
  // emphasis ''
  logger.debug( "''" );
  if( em ){
    em = false;
    return( "</em>" );
  }
  else{
    em = true;
    return( "<em>" );
  }
}

<NORMAL>"::"	{
  // center ::
  logger.debug( "::" );
  if( center ){
    center = false;
    return( "</div>" );
  }
  else{
    center = true;
    return( "<div align=\"center\">" );
  }
}

<NORMAL>"==="	{
  // underline ===
  logger.debug( "===" );
  if( underline ){
    underline = false;
    return( "</u>" );
  }
  else{
    underline = true;
    return( "<u>" );
  }
}

<NORMAL>"{{{" {
  // code start {{{
  isCode = true;
  return "<code>";
}

<NORMAL>"}}}" {
  // code end }}}
  isCode=false;
  return "</code>";
}

<NORMAL>" " {
  // space
  if (isCode)
    return "&nbsp;";
  else
    return " ";
}

<NORMAL>"@@" {
  // Line break @@
  return "<br/>";
}

<NORMAL>"@~@" {
  // Page break @~@
  return "<p style=\"page-break-after:always\">&nbsp;</p>";
}

<NORMAL>"\\." {
  // special space \;
	return "&nbsp;";
}

<NORMAL>"\\;" {
  // special tab \;
	return "&nbsp;&nbsp;&nbsp;";
}


<NORMAL, OFF, PRE>{entity}  {
  return yytext();
}
<NORMAL, OFF, PRE>"&lt;"  {
  return "&amp;lt;";
}
<NORMAL, OFF, PRE>"&gt;" {
  return "&amp;gt;";
}
<NORMAL, OFF, PRE>"&amp;"	{
  return "&amp;amp;";
}
<NORMAL, OFF, PRE>"<"	{
  return "&lt;";
}
<NORMAL, OFF, PRE>">" {
  return "&gt;";
}
<NORMAL, OFF, PRE>"&"	{
  return "&amp;";
}

<PRE>{newline}{newline} {
  logger.debug( "{newline}x2 leaving pre" );
	yybegin( NORMAL );
  return yytext();
}

<NORMAL, OFF>{newline} {
  logger.debug( "{newline}" );
  if( h1 ){
    h1 = false;
    return( "</h1>" );
  }
  if( h2 ){
    h2 = false;
    return( "</h2>" );
  }
  if( h3 ){
    h3 = false;
    return( "</h3>" );
  }
  return yytext();
}

<NORMAL>(@@@@{newline}) {
  logger.debug( "@@@@{newline} entering PRE" );
  yybegin( PRE );
  return yytext();
}

<NORMAL, OFF, EXTERNAL>{whitespace} {
  logger.debug( "{whitespace}" );
  return yytext();
}

<PRE>{whitespace} {
  logger.debug( "PRE {whitespace}" );
  return yytext();
}

<NORMAL, PRE, OFF, EXTERNAL>. {
  logger.debug( ". (" + yytext() + ")" );
  return yytext();
}

<NORMAL>{colorstart} {
  logger.debug( "color start" );
  
  StringBuffer sb = new StringBuffer() ;
  if( color ){
    sb.append( "</font>" );
  }  
  color = true;
  sb.append( "<font color=\"")
    .append( yytext().substring( 1,yytext().length()-1) )
    .append( "\">" );
  
  return sb.toString();
}

<NORMAL>{colorend} {
  if( color ){
    logger.debug( "color end" );
  
    color = false ;
    return( "</font>" );
  }
  else {
    return yytext();
  }
  
}
