/*
Very Quick Wiki - WikiWikiWeb clone
Copyright (C) 2001-2002 Gareth Cronin

This program is free software; you can redistribute it and/or modify
it under the terms of the latest version of the GNU Lesser General
Public License as published by the Free Software Foundation;

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program (gpl.txt); if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
package vqwiki.file;

import JSX.ObjIn;
import JSX.ObjOut;
import org.apache.log4j.Logger;
import vqwiki.*;
import vqwiki.utils.JSPUtils;
import vqwiki.utils.Utilities;

import javax.servlet.http.HttpServletRequest;
import java.io.*;
import java.util.*;

public class FileChangeLog implements ChangeLog {

  private static final Logger logger = Logger.getLogger(FileChangeLog.class);

  private static FileChangeLog instance;

  // recent changes file
  public static final String RECENTFILE = "recent.xml";
  private HashMap cache;

  private FileChangeLog() {
    cache = new HashMap();
  }

  private Hashtable getTableFor(String virtualWiki) throws IOException, ClassNotFoundException {
    if (virtualWiki == null) virtualWiki = "";
    if (cache.containsKey(virtualWiki))
      return (Hashtable) cache.get(virtualWiki);
    File changesPath = FileHandler.getPathFor(virtualWiki, RECENTFILE);
    logger.debug("reading changes for " + virtualWiki + " from " + changesPath);
    Hashtable vwikiCache;
    if (!changesPath.exists()) {
      logger.debug("no file exists, creating now");
      vwikiCache = new Hashtable();
      cache.put(virtualWiki, vwikiCache);
      saveTableFor(virtualWiki);
      return vwikiCache;
    }
    logger.debug("reading file from disk");
    FileInputStream fileInputStream = new FileInputStream(changesPath);
    ObjIn in = new ObjIn(fileInputStream);
    vwikiCache = (Hashtable) in.readObject();
    cache.put(virtualWiki, vwikiCache);
    in.close();
    fileInputStream.close();
    return vwikiCache;
  }

  private void saveTableFor(String virtualWiki) throws IOException {
    if (virtualWiki == null) virtualWiki = "";
    File changesPath = FileHandler.getPathFor(virtualWiki, RECENTFILE);
    FileOutputStream fileOutputStream = new FileOutputStream(changesPath);
    ObjOut out = new ObjOut(fileOutputStream);
    Object changeTable = cache.get(virtualWiki);
    logger.debug("Change table for " + virtualWiki + ": " + changeTable);
    out.writeObject(changeTable);
    out.close();
    fileOutputStream.close();
    logger.debug("wrote changes to disk: " + changesPath);
  }

  public static final FileChangeLog getInstance() {
    if (instance == null)
      instance = new FileChangeLog();
    return instance;
  }

  public synchronized void logChange(Change change, HttpServletRequest request)
      throws IOException, FileNotFoundException, ClassNotFoundException {

    logger.debug("logging change " + change);

    String date = Utilities.formatDate(new Date());

    String virtualWiki = change.getVirtualWiki();
    Hashtable changesTable = getTableFor(virtualWiki);

    Vector changesInFile;
    if (changesTable.get(date) == null) {
      logger.debug("no changes on " + date);
      changesInFile = new Vector();
      changesTable.put(date, changesInFile);
    }
    else {
      logger.debug("adding to changes for " + date);
      changesInFile = (Vector) changesTable.get(date);
    }

    boolean suppressNotifyInSameDay =
        Environment.getInstance().getBooleanSetting(Environment.PROPERTY_SUPPRESS_NOTIFY_WITHIN_SAME_DAY);

    logger.debug("changesInFile: " + changesInFile);
    boolean changedToday = false;
    if (changesInFile.contains(change)) {
      if (logger.isDebugEnabled()) {
        logger.debug("removing existing change");
        if (suppressNotifyInSameDay){
          logger.debug("not sending notifications because the topic has already been changed today");
        }
      }
      changesInFile.remove(change);
      changedToday = true;
    }


    if (!changedToday && suppressNotifyInSameDay
        || !suppressNotifyInSameDay) {
      try {
        logger.debug("running notifier");
        Notify notifier = new FileNotify(virtualWiki, change.getTopic());
        notifier.sendNotifications(JSPUtils.createRootPath(request, virtualWiki, true), request.getLocale());
      }
      catch (Exception e) {
        logger.warn("exception occurred in notifier", e);
        e.printStackTrace();
      }
    }
    // add change to front of Vector, so that date-sorting within day works correctly.
    logger.debug("adding change " + change);
    changesInFile.add(0, change);


    saveTableFor(virtualWiki);
  }

  /**
   * Get all recent changes for a particular date
   * @param virtualWiki the virtual wiki
   * @param d the date of the changes
   * @return
   * @throws IOException
   * @throws FileNotFoundException
   * @throws ClassNotFoundException
   */
  public Collection getChanges(String virtualWiki, Date d)
      throws IOException, FileNotFoundException, ClassNotFoundException {
    String date = Utilities.formatDate(d);
    return (Collection) getTableFor(virtualWiki).get(date);
  }

  /**
   * Delete the existing recent changes file
   * @param virtualWiki
   */
  public void deleteChangeTableFile(String virtualWiki) {
    try {
      FileHandler.getPathFor(virtualWiki, RECENTFILE).delete();
    }
    catch (Exception err) {
      logger.error("Unable to delete recent changes file for " + virtualWiki, err);
    }
  }
}
